<?php
// api.php - Backend JSON para Ventas Blanquita (robusto + notificaciones)
// PHP 8.1+

declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');

ini_set('display_errors', '0');
ini_set('display_startup_errors', '0');
error_reporting(E_ALL);

require_once __DIR__ . '/includes/db.php';
require_once __DIR__ . '/includes/auth.php';

// WhatsApp (Twilio)
$waHelper = __DIR__ . '/whatsapp_helper.php';
if (is_file($waHelper)) require_once $waHelper;
$waOutbox = __DIR__ . '/includes/wa_outbox.php';
if (is_file($waOutbox)) require_once $waOutbox;

// SMS (Twilio)
$smsHelper = __DIR__ . '/sms_helper.php';
if (is_file($smsHelper)) require_once $smsHelper;
$smsOutbox = __DIR__ . '/includes/sms_outbox.php';
if (is_file($smsOutbox)) require_once $smsOutbox;

// Push (opcional)
$pushHelpers = __DIR__ . '/push_helper.php';
if (is_file($pushHelpers)) require_once $pushHelpers;

$jsonOpts = JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES;

// Config / Debug
$CFG   = function_exists('app_config') ? (app_config() ?: []) : [];
$DEBUG = (bool)($CFG['debug'] ?? false);
$DEBUG = $DEBUG || (trim((string)getenv('VB_APP_DEBUG')) === '1');

// Toggles de notificación (por defecto: WhatsApp ON, SMS OFF)
$WA_ENABLED  = true;
$SMS_ENABLED = false;

$envWa = trim((string)getenv('VB_WA_ENABLED'));
if ($envWa !== '') $WA_ENABLED = ($envWa === '1' || strtolower($envWa) === 'true');

$envSms = trim((string)getenv('VB_SMS_ENABLED'));
if ($envSms !== '') $SMS_ENABLED = ($envSms === '1' || strtolower($envSms) === 'true');

if (array_key_exists('wa_enabled', $CFG))  $WA_ENABLED  = (bool)$CFG['wa_enabled'];
if (array_key_exists('sms_enabled', $CFG)) $SMS_ENABLED = (bool)$CFG['sms_enabled'];

// Acción
$action = trim((string)($_GET['action'] ?? ''));
if ($action === '') $action = trim((string)($_POST['action'] ?? ''));

function out_json(array $data, int $jsonOpts, int $httpCode = 200): void {
    http_response_code($httpCode);
    echo json_encode($data, $jsonOpts);
    exit;
}

function require_roles(array $allowed, string $role, int $jsonOpts): void {
    if (!in_array($role, $allowed, true)) {
        out_json(['ok' => false, 'error' => 'FORBIDDEN'], $jsonOpts, 403);
    }
}

function read_input(): array {
    $ct = $_SERVER['CONTENT_TYPE'] ?? $_SERVER['HTTP_CONTENT_TYPE'] ?? '';
    if (stripos($ct, 'application/json') !== false) {
        $raw = file_get_contents('php://input');
        $j = json_decode($raw ?: '', true);
        return is_array($j) ? $j : [];
    }
    return is_array($_POST) ? $_POST : [];
}

function push_try(callable $fn, bool $debug = false): void {
    try { $fn(); } catch (Throwable $e) { if ($debug) error_log('[PUSH] '.$e->getMessage()); }
}

function wa_try(callable $fn, bool $debug = false): mixed {
    try { return $fn(); } catch (Throwable $e) { if ($debug) error_log('[WA] '.$e->getMessage()); return null; }
}

function sms_try(callable $fn, bool $debug = false): mixed {
    try { return $fn(); } catch (Throwable $e) { if ($debug) error_log('[SMS] '.$e->getMessage()); return null; }
}

function wa_try_process(PDO $pdo, bool $debug = false): void {
    if (function_exists('wa_process_outbox')) {
        try { wa_process_outbox($pdo, 10); } catch (Throwable $e) { if ($debug) error_log('[WA_WORKER] '.$e->getMessage()); }
    }
}

function sms_try_process(PDO $pdo, bool $debug = false): void {
    if (function_exists('sms_process_outbox')) {
        try { sms_process_outbox($pdo, 10); } catch (Throwable $e) { if ($debug) error_log('[SMS_WORKER] '.$e->getMessage()); }
    }
}

function table_has_column(PDO $pdo, string $table, string $column): bool {
    static $cache = [];
    $key = strtolower($table.'.'.$column);
    if (isset($cache[$key])) return (bool)$cache[$key];

    $stmt = $pdo->prepare("
        SELECT COUNT(*)
        FROM information_schema.columns
        WHERE table_schema = DATABASE()
          AND table_name = ?
          AND column_name = ?
    ");
    $stmt->execute([$table, $column]);
    $ok = ((int)$stmt->fetchColumn() > 0);
    $cache[$key] = $ok;
    return $ok;
}

function pick_column(PDO $pdo, string $table, array $candidates): ?string {
    foreach ($candidates as $c) if (table_has_column($pdo, $table, $c)) return $c;
    return null;
}

function insert_dynamic(PDO $pdo, string $table, array $data): int {
    $cols = array_keys($data);
    $ph   = array_fill(0, count($cols), '?');
    $sql  = "INSERT INTO {$table} (" . implode(',', $cols) . ") VALUES (" . implode(',', $ph) . ")";
    $stmt = $pdo->prepare($sql);
    $stmt->execute(array_values($data));
    return (int)$pdo->lastInsertId();
}

function update_dynamic_by_id(PDO $pdo, string $table, int $id, array $data): void {
    if (!$data) return;
    $sets = [];
    $vals = [];
    foreach ($data as $k => $v) { $sets[] = "{$k} = ?"; $vals[] = $v; }
    $vals[] = $id;
    $sql = "UPDATE {$table} SET " . implode(', ', $sets) . " WHERE id = ?";
    $stmt = $pdo->prepare($sql);
    $stmt->execute($vals);
}

function items_desc(array $items): string {
    $txt = '';
    foreach ($items as $it) {
        $n = (string)($it['name'] ?? $it['producto'] ?? '');
        $q = (string)($it['qty'] ?? $it['quantity'] ?? '');
        if ($n !== '' && $q !== '') $txt .= "- {$n}: {$q} piezas\n";
    }
    return $txt;
}

function wa_link_from_phone(string $phoneRawOrE164, string $message): ?string {
    if (function_exists('wa_make_link')) {
        $lnk = (string)wa_make_link($phoneRawOrE164, $message);
        return $lnk !== '' ? $lnk : null;
    }
    $digits = preg_replace('/\D+/', '', $phoneRawOrE164) ?? '';
    if ($digits === '') return null;
    return "https://wa.me/{$digits}?text=" . rawurlencode($message);
}

try {
    $pdo = pdo();
} catch (Throwable $e) {
    $out = ['ok' => false, 'error' => 'DB_ERROR'];
    if ($DEBUG) $out['detail'] = $e->getMessage();
    out_json($out, $jsonOpts, 500);
}

$u = current_user();

// Acciones públicas (sin sesión)
$accionesPublicas = ['ping', 'push_public_key'];

if (!$u && !in_array($action, $accionesPublicas, true)) {
    out_json(['ok' => false, 'error' => 'NO_AUTH'], $jsonOpts, 401);
}

$userId = $u ? (int)($u['id'] ?? 0) : 0;
$role   = $u ? (string)($u['role'] ?? '') : '';

try {
    switch ($action) {

        case 'ping':
            out_json(['ok' => true, 'time' => date('Y-m-d H:i:s')], $jsonOpts);

        case 'push_public_key': {
            $pub = function_exists('getVapidPublicKey')
                ? getVapidPublicKey()
                : (defined('VAPID_PUBLIC_KEY') ? VAPID_PUBLIC_KEY : '');
            out_json(['ok' => true, 'publicKey' => (string)$pub], $jsonOpts);
        }

        /* ================== PUSH: SUSCRIPCIÓN ================== */
        case 'push_subscribe': {
            if ($_SERVER['REQUEST_METHOD'] !== 'POST') out_json(['ok'=>false,'error'=>'METHOD_NOT_ALLOWED'], $jsonOpts, 405);
            if (!$userId) out_json(['ok'=>false,'error'=>'NO_AUTH'], $jsonOpts, 401);

            $input = read_input();
            if (isset($input['subscription']) && is_array($input['subscription'])) $input = $input['subscription'];

            $endpoint = trim((string)($input['endpoint'] ?? ''));
            $keys     = $input['keys'] ?? [];
            $p256dh   = is_array($keys) ? trim((string)($keys['p256dh'] ?? '')) : '';
            $authKey  = is_array($keys) ? trim((string)($keys['auth'] ?? '')) : '';

            if ($endpoint === '' || $p256dh === '' || $authKey === '') {
                out_json(['ok'=>false,'error'=>'SUBSCRIPTION_INCOMPLETE'], $jsonOpts, 400);
            }

            $stmt = $pdo->prepare("
                INSERT INTO push_subscriptions (user_id, endpoint, p256dh, auth, created_at, last_used)
                VALUES (?, ?, ?, ?, NOW(), NULL)
                ON DUPLICATE KEY UPDATE
                    p256dh    = VALUES(p256dh),
                    auth      = VALUES(auth),
                    last_used = NOW()
            ");
            $stmt->execute([$userId, $endpoint, $p256dh, $authKey]);

            out_json(['ok' => true], $jsonOpts);
        }

        /* ================== PUSH: ESTATUS (DIAGNÓSTICO) ================== */
        case 'push_status': {
            if (!$userId) out_json(['ok'=>false,'error'=>'NO_AUTH'], $jsonOpts, 401);

            $stmt = $pdo->prepare("
                SELECT id, endpoint, created_at, last_used
                FROM push_subscriptions
                WHERE user_id = ?
                ORDER BY id DESC
                LIMIT 10
            ");
            $stmt->execute([$userId]);
            $rows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];

            foreach ($rows as &$r) {
                $r['endpoint'] = substr((string)$r['endpoint'], 0, 80) . '...';
            }
            unset($r);

            out_json([
                'ok' => true,
                'user_id' => (int)$userId,
                'role' => (string)$role,
                'count' => count($rows),
                'rows' => $rows
            ], $jsonOpts);
        }

        /* ================== DIRECCIONES CLIENTE ================== */
        case 'direcciones_listar': {
            require_roles(['cliente'], $role, $jsonOpts);

            $stmt = $pdo->prepare("
                SELECT id, label, address, postal_code
                FROM customer_addresses
                WHERE user_id = ?
                ORDER BY created_at DESC, id DESC
            ");
            $stmt->execute([$userId]);
            out_json(['ok' => true, 'rows' => $stmt->fetchAll(PDO::FETCH_ASSOC)], $jsonOpts);
        }

        case 'direccion_crear': {
            require_roles(['cliente'], $role, $jsonOpts);
            if ($_SERVER['REQUEST_METHOD'] !== 'POST') out_json(['ok'=>false,'error'=>'METHOD_NOT_ALLOWED'], $jsonOpts, 405);

            $input = read_input();
            $label = trim((string)($input['label'] ?? ''));
            $addr  = trim((string)($input['address'] ?? ''));
            $cp    = trim((string)($input['postal_code'] ?? ''));

            if ($addr === '') out_json(['ok'=>false,'error'=>'ADDRESS_REQUIRED'], $jsonOpts, 400);

            $stmt = $pdo->prepare("
                INSERT INTO customer_addresses (user_id, label, address, postal_code)
                VALUES (?, ?, ?, ?)
            ");
            $stmt->execute([$userId, $label, $addr, $cp]);

            out_json(['ok' => true, 'id' => (int)$pdo->lastInsertId()], $jsonOpts);
        }

        /* ================== CREAR PEDIDO ================== */
        case 'crear_pedido': {
            require_roles(['cliente'], $role, $jsonOpts);
            if ($_SERVER['REQUEST_METHOD'] !== 'POST') out_json(['ok'=>false,'error'=>'METHOD_NOT_ALLOWED'], $jsonOpts, 405);

            $input = read_input();

            $items            = $input['items'] ?? [];
            $delivery_date    = trim((string)($input['delivery_date'] ?? ''));
            $delivery_time    = trim((string)($input['delivery_time'] ?? ''));
            $delivery_date = trim((string)($input['delivery_date'] ?? ''));
            $itemsInput    = $input['items'] ?? null;

            if ($delivery_time !== '' && !preg_match('/^\d{2}:\d{2}$/', $delivery_time)) {
                out_json(['ok'=>false,'error'=>'DELIVERY_TIME_INVALID'], $jsonOpts, 400);
            }
            if ($delivery_date !== '' && !preg_match('/^\d{4}-\d{2}-\d{2}$/', $delivery_date)) {
                out_json(['ok'=>false,'error'=>'DELIVERY_DATE_INVALID'], $jsonOpts, 400);
            }

            $delivery_address = trim((string)($input['delivery_address'] ?? ''));
            $delivery_label   = trim((string)($input['delivery_label'] ?? ''));
            $instructions     = trim((string)($input['instructions'] ?? ''));
            $comments         = trim((string)($input['comments'] ?? ''));

            if (!is_array($items) || !count($items)) out_json(['ok'=>false,'error'=>'ITEMS_REQUIRED'], $jsonOpts, 400);
            if ($delivery_date === '') out_json(['ok'=>false,'error'=>'DELIVERY_DATE_REQUIRED'], $jsonOpts, 400);
            if ($delivery_address === '') out_json(['ok'=>false,'error'=>'DELIVERY_ADDRESS_REQUIRED'], $jsonOpts, 400);

            // ================= Reglas de entrega (server-side) =================
            // - Si la entrega es "hoy", debe ser mínimo NOW + 6 horas.
            // - Si el pedido se crea después de las 18:00 (hora CDMX), no se permite seleccionar el mismo día;
            //   la entrega mínima es al día siguiente a las 06:00.
            // - Para fechas futuras, la hora mínima es 06:00.
            $tzMX  = new DateTimeZone('America/Mexico_City');
            $nowMX = new DateTime('now', $tzMX);
            $cutMX = (clone $nowMX)->setTime(18, 0, 0);
            $today = $nowMX->format('Y-m-d');
            $minDate = ($nowMX >= $cutMX) ? (clone $nowMX)->modify('+1 day')->format('Y-m-d') : $today;

            if ($delivery_date < $minDate) {
                out_json(['ok'=>false,'error'=>'DELIVERY_DATE_TOO_SOON','min_date'=>$minDate], $jsonOpts, 400);
            }

            // Normaliza / valida formato HH:MM (si viene vacío lo asignamos a la hora mínima)
            if ($delivery_time !== '' && !preg_match('/^\d{2}:\d{2}$/', $delivery_time)) {
                out_json(['ok'=>false,'error'=>'DELIVERY_TIME_INVALID'], $jsonOpts, 400);
            }

            $minDT = null;
            if ($delivery_date === $today && $nowMX < $cutMX) {
                $minDT = (clone $nowMX)->modify('+6 hours');
            } else {
                $minDT = new DateTime($delivery_date . ' 06:00:00', $tzMX);
            }

            $minTimeStr = $minDT->format('H:i');
            if ($delivery_time === '') {
                $delivery_time = $minTimeStr;
            } else {
                $selDT = new DateTime($delivery_date . ' ' . $delivery_time . ':00', $tzMX);
                if ($selDT < $minDT) {
                    out_json(['ok'=>false,'error'=>'DELIVERY_TIME_TOO_EARLY','min_time'=>$minTimeStr], $jsonOpts, 400);
                }
            }

            $cleanItems = [];
            foreach ($items as $it) {
                $pid = (int)($it['product_id'] ?? 0);
                $qty = (float)($it['qty'] ?? 0);
                if ($pid > 0 && $qty > 0) $cleanItems[] = ['product_id' => $pid, 'qty' => $qty];
            }
            if (!$cleanItems) out_json(['ok'=>false,'error'=>'NO_VALID_ITEMS'], $jsonOpts, 400);

            $ids = array_values(array_unique(array_column($cleanItems, 'product_id')));
            $in  = implode(',', array_fill(0, count($ids), '?'));

            $stmt = $pdo->prepare("SELECT id, name, price FROM products WHERE id IN ($in)");
            $stmt->execute($ids);
            $prodRows = $stmt->fetchAll(PDO::FETCH_ASSOC);

            $mapProd = [];
            foreach ($prodRows as $pr) $mapProd[(int)$pr['id']] = $pr;

            $totalQty  = 0.0;
            $totalMXN  = 0.0;
            $itemsFull = [];

            foreach ($cleanItems as $it) {
                $pid = (int)$it['product_id'];
                $qty = (float)$it['qty'];
                $p   = $mapProd[$pid] ?? null;
                if (!$p) continue;

                $unit = (float)($p['price'] ?? 0);
                $line = $unit * $qty;

                $totalQty += $qty;
                $totalMXN += $line;

                $itemsFull[] = [
                    'product_id' => $pid,
                    'qty'        => $qty,
                    'unit_price' => $unit,
                    'name'       => (string)($p['name'] ?? ''),
                ];
            }

            if (!$itemsFull) out_json(['ok'=>false,'error'=>'ITEMS_WITHOUT_PRICE'], $jsonOpts, 400);

            $pdo->beginTransaction();
            try {
                $status = 'pendiente';

                $orderData = [
                    'customer_id'      => $userId,
                    'delivery_address' => $delivery_address,
                    'delivery_date'    => $delivery_date,
                    'status'           => $status,
                ];

                if (table_has_column($pdo, 'orders', 'delivery_time')) $orderData['delivery_time'] = ($delivery_time !== '') ? $delivery_time : null;
                if (table_has_column($pdo, 'orders', 'delivery_label')) $orderData['delivery_label'] = ($delivery_label !== '') ? $delivery_label : null;
                if (table_has_column($pdo, 'orders', 'delivery_instructions')) $orderData['delivery_instructions'] = ($instructions !== '') ? $instructions : null;
                if (table_has_column($pdo, 'orders', 'comments')) $orderData['comments'] = ($comments !== '') ? $comments : null;
                if (table_has_column($pdo, 'orders', 'total')) $orderData['total'] = $totalMXN;

                $qtyCol = pick_column($pdo, 'orders', ['Qty_pza', 'qty_pza', 'qty']);
                if ($qtyCol) $orderData[$qtyCol] = $totalQty;

                // created_at puede tener default; esto es opcional
                if (table_has_column($pdo, 'orders', 'created_at')) $orderData['created_at'] = date('Y-m-d H:i:s');

                $orderId = insert_dynamic($pdo, 'orders', $orderData);

                $stmtItem = $pdo->prepare("
                    INSERT INTO order_items (order_id, product_id, quantity, unit_price)
                    VALUES (?, ?, ?, ?)
                ");
                foreach ($itemsFull as $it) {
                    $stmtItem->execute([$orderId, $it['product_id'], $it['qty'], $it['unit_price']]);
                }

                $pdo->commit();
            } catch (Throwable $e) {
                $pdo->rollBack();
                throw $e;
            }

            // Notificaciones (NO deben tumbar el pedido)
            $clienteNombre = (string)($u['name'] ?? '');
            $clienteTel    = (string)($u['phone'] ?? '');

            // Push internos
            if (function_exists('sendPushToRole')) {
                $title = 'Nuevo pedido';
                $body  = 'Pedido #' . $orderId . ' · Piezas: ' . $totalQty . ' · Entrega: ' . $delivery_date;
                push_try(fn() => sendPushToRole($pdo, 'ventas', $title, $body, ['url' => './?order=' . $orderId]), $DEBUG);
                push_try(fn() => sendPushToRole($pdo, 'admin',  $title, $body, ['url' => './?order=' . $orderId]), $DEBUG);
                push_try(fn() => sendPushToRole($pdo, 'tienda', $title, $body, ['url' => './?order=' . $orderId]), $DEBUG);
            }
            if (function_exists('sendPushToUser') && $u) {
                $title = 'Pedido recibido';
                $body  = 'Tu pedido #' . $orderId . ' fue registrado. Te avisaremos cuando se autorice.';
                push_try(fn() => sendPushToUser($pdo, (int)$u['id'], $title, $body, ['url' => './']), $DEBUG);
            }

            $lineaHora  = $delivery_time ? "Hora: {$delivery_time}\n" : '';
            $lineaInst  = $instructions ? "Instrucciones: {$instructions}\n" : '';
            $lineaComm  = $comments ? "Comentarios: {$comments}\n" : '';
            $lineaLugar = $delivery_label ? "Lugar: {$delivery_label}\n" : '';

            $txtItems = items_desc($itemsFull);

            $mensajeVentas = "NUEVO PEDIDO #{$orderId}\n"
                . "Cliente: {$clienteNombre}\n"
                . ($clienteTel ? "Teléfono: {$clienteTel}\n" : '')
                . "Entrega: {$delivery_date}\n"
                . $lineaHora
                . $lineaLugar
                . "Dirección: {$delivery_address}\n"
                . $lineaInst
                . $lineaComm
                . "\nArtículos:\n"
                . ($txtItems ?: "- Sin detalle de artículos\n");

            $wa_result = null;
            if ($WA_ENABLED && function_exists('wa_queue_to_role')) {
                $wa_result = wa_try(fn() => wa_queue_to_role($pdo, 'ventas', 'order_created', (int)$orderId, $mensajeVentas), $DEBUG);
                wa_try_process($pdo, $DEBUG);
            }

            $sms_result = ['ventas' => null, 'cliente' => null];
            if ($SMS_ENABLED && function_exists('sms_queue_to_role')) {
                $smsVentas = "[Ventas Blanquita] NUEVO pedido #{$orderId}. Entrega: {$delivery_date}" . ($delivery_time ? " {$delivery_time}" : "") . ". Cliente: {$clienteNombre}.";
                $sms_result['ventas'] = sms_try(fn() => sms_queue_to_role($pdo, 'ventas', 'order_created', (int)$orderId, $smsVentas), $DEBUG);
                sms_try_process($pdo, $DEBUG);
            }
            if ($SMS_ENABLED && function_exists('sms_queue_to_phone') && $u) {
                $countryHint = (string)($u['phone_country'] ?? 'MX');
                $smsCliente = "[Ventas Blanquita] Recibimos tu pedido #{$orderId}. Entrega: {$delivery_date}" . ($delivery_time ? " {$delivery_time}" : "") . ".";
                $sms_result['cliente'] = sms_try(fn() => sms_queue_to_phone($pdo, (string)($u['phone'] ?? ''), 'order_created_customer', (int)$orderId, $smsCliente, $countryHint), $DEBUG);
                sms_try_process($pdo, $DEBUG);
            }

            // Fallback manual por link (a número de tienda/ventas)
            $waLink = null;
            $storePhone = trim((string)($CFG['store_phone'] ?? ''));
            if ($storePhone !== '') $waLink = wa_link_from_phone($storePhone, $mensajeVentas);

            out_json(['ok' => true, 'order_id' => (int)$orderId, 'wa' => $wa_result, 'wa_link' => $waLink, 'sms' => $sms_result], $jsonOpts);
        }

        /* ================== MIS PEDIDOS ================== */
        case 'mis_pedidos': {
            require_roles(['cliente'], $role, $jsonOpts);

            $hasCreated = table_has_column($pdo, 'orders', 'created_at');
            $orderBy = $hasCreated ? "o.created_at DESC" : "o.id DESC";

            $stmt = $pdo->prepare("
                SELECT
                    o.id,
                    o.status,
                    o.delivery_date,
                    SUM(oi.quantity) AS qty
                FROM orders o
                LEFT JOIN order_items oi ON oi.order_id = o.id
                WHERE o.customer_id = ?
                GROUP BY o.id, o.status, o.delivery_date
                ORDER BY {$orderBy}
                LIMIT 5
            ");
            $stmt->execute([$userId]);
            out_json(['ok' => true, 'rows' => $stmt->fetchAll(PDO::FETCH_ASSOC)], $jsonOpts);
        }

        /* ================== CANCELAR PEDIDO ================== */
        case 'cancelar_pedido': {
            require_roles(['cliente'], $role, $jsonOpts);
            if ($_SERVER['REQUEST_METHOD'] !== 'POST') out_json(['ok'=>false,'error'=>'METHOD_NOT_ALLOWED'], $jsonOpts, 405);

            $input   = read_input();
            $orderId = (int)($input['order_id'] ?? 0);
            if ($orderId <= 0) out_json(['ok'=>false,'error'=>'INVALID_ORDER_ID'], $jsonOpts, 400);

            $selTime  = table_has_column($pdo, 'orders', 'delivery_time') ? "o.delivery_time" : "NULL AS delivery_time";
            $selCountry = table_has_column($pdo, 'users', 'phone_country') ? "u.phone_country AS phone_country" : "'MX' AS phone_country";

            $stmt = $pdo->prepare("
                SELECT o.id, o.customer_id, o.status, o.delivery_date, {$selTime},
                       o.delivery_address,
                       u.name AS cliente, u.phone AS phone, {$selCountry}
                FROM orders o
                JOIN users u ON u.id = o.customer_id
                WHERE o.id = ? AND o.customer_id = ?
                LIMIT 1
            ");
            $stmt->execute([$orderId, $userId]);
            $pedido = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$pedido) out_json(['ok'=>false,'error'=>'ORDER_NOT_FOUND'], $jsonOpts, 404);
            if ((string)$pedido['status'] !== 'pendiente') out_json(['ok'=>false,'error'=>'ONLY_PENDING_CAN_CANCEL'], $jsonOpts, 400);

            $it = $pdo->prepare("
                SELECT oi.quantity, p.name AS producto
                FROM order_items oi
                JOIN products p ON p.id = oi.product_id
                WHERE oi.order_id = ?
            ");
            $it->execute([$orderId]);
            $items = $it->fetchAll(PDO::FETCH_ASSOC) ?: [];

            $upd = ['status' => 'cancelada'];
            if (table_has_column($pdo, 'orders', 'updated_at')) $upd['updated_at'] = date('Y-m-d H:i:s');
            $pdo->beginTransaction();
            try {
                            $pdo->beginTransaction();
            try {
                update_dynamic_by_id($pdo, 'orders', $orderId, $upd);

                // Actualizar cantidades por artículo (solo productos ya existentes en el pedido)
                if (is_array($itemsInput)) {
                    $cur = $pdo->prepare("SELECT product_id FROM order_items WHERE order_id = ?");
                    $cur->execute([$orderId]);
                    $existing = $cur->fetchAll(PDO::FETCH_ASSOC) ?: [];
                    $existingIds = [];
                    foreach ($existing as $r) $existingIds[(int)$r['product_id']] = true;

                    $clean = [];
                    foreach ($itemsInput as $it) {
                        if (!is_array($it)) continue;
                        $pid = (int)($it['product_id'] ?? 0);
                        $qty = (float)($it['qty'] ?? 0);
                        if ($pid > 0 && isset($existingIds[$pid])) {
                            // qty <= 0 => eliminar línea
                            $clean[$pid] = $qty;
                        }
                    }

                    if ($clean) {
                        $stUpd = $pdo->prepare("UPDATE order_items SET quantity = ? WHERE order_id = ? AND product_id = ?");
                        $stDel = $pdo->prepare("DELETE FROM order_items WHERE order_id = ? AND product_id = ?");
                        foreach ($clean as $pid => $qty) {
                            if ($qty <= 0) {
                                $stDel->execute([$orderId, (int)$pid]);
                            } else {
                                $stUpd->execute([(float)$qty, $orderId, (int)$pid]);
                            }
                        }

                        $cnt = $pdo->prepare("SELECT COUNT(*) FROM order_items WHERE order_id = ?");
                        $cnt->execute([$orderId]);
                        if ((int)$cnt->fetchColumn() <= 0) {
                            out_json(['ok'=>false,'error'=>'AT_LEAST_ONE_ITEM_REQUIRED'], $jsonOpts, 400);
                        }

                        // Recalcular totales
                        $sum = $pdo->prepare("SELECT SUM(quantity) AS qty, SUM(quantity * unit_price) AS total FROM order_items WHERE order_id = ?");
                        $sum->execute([$orderId]);
                        $tot = $sum->fetch(PDO::FETCH_ASSOC) ?: ['qty' => 0, 'total' => 0];

                        $upd2 = [];
                        $qtyCol = pick_column($pdo, 'orders', ['Qty_pza', 'qty_pza', 'qty']);
                        if ($qtyCol) $upd2[$qtyCol] = (float)($tot['qty'] ?? 0);
                        if (table_has_column($pdo, 'orders', 'total')) $upd2['total'] = (float)($tot['total'] ?? 0);
                        if ($upd2) update_dynamic_by_id($pdo, 'orders', $orderId, $upd2);
                    }
                }

                $pdo->commit();
            } catch (Throwable $e) {
                if ($pdo->inTransaction()) $pdo->rollBack();
                throw $e;
            }


                // ======= Actualización de cantidades por artículo (opcional) =======
                // Nota: Para no romper lógica existente, esto solo corre si se envía un array "items".
                // Se permiten ajustes únicamente de productos ya existentes en el pedido.
                if (is_array($itemsInput)) {
                    // Productos existentes en el pedido
                    $cur = $pdo->prepare("SELECT product_id FROM order_items WHERE order_id = ?");
                    $cur->execute([$orderId]);
                    $existing = $cur->fetchAll(PDO::FETCH_ASSOC) ?: [];
                    $existingIds = [];
                    foreach ($existing as $r) $existingIds[(int)$r['product_id']] = true;

                    $clean = [];
                    foreach ($itemsInput as $it) {
                        if (!is_array($it)) continue;
                        $pid = (int)($it['product_id'] ?? 0);
                        $qty = (float)($it['qty'] ?? 0);
                        if ($pid <= 0) continue;
                        if (!isset($existingIds[$pid])) continue; // no agregar productos nuevos
                        // qty <= 0 => eliminar
                        $clean[$pid] = $qty;
                    }

                    if ($clean) {
                        $updItem = $pdo->prepare("UPDATE order_items SET quantity = ? WHERE order_id = ? AND product_id = ?");
                        $delItem = $pdo->prepare("DELETE FROM order_items WHERE order_id = ? AND product_id = ?");
                        foreach ($clean as $pid => $qty) {
                            if ($qty <= 0) {
                                $delItem->execute([$orderId, (int)$pid]);
                            } else {
                                $updItem->execute([$qty, $orderId, (int)$pid]);
                            }
                        }

                        // Validar que quede al menos 1 artículo
                        $cnt = $pdo->prepare("SELECT COUNT(*) FROM order_items WHERE order_id = ?");
                        $cnt->execute([$orderId]);
                        $n = (int)$cnt->fetchColumn();
                        if ($n <= 0) {
                            out_json(['ok'=>false,'error'=>'ORDER_MUST_HAVE_ITEMS'], $jsonOpts, 400);
                        }

                        // Recalcular totales y actualizar orden (si existen columnas)
                        $sum = $pdo->prepare("SELECT SUM(quantity) AS qty, SUM(quantity*unit_price) AS total FROM order_items WHERE order_id = ?");
                        $sum->execute([$orderId]);
                        $sr = $sum->fetch(PDO::FETCH_ASSOC) ?: ['qty' => 0, 'total' => 0];

                        $upd2 = [];
                        $qtyCol = pick_column($pdo, 'orders', ['Qty_pza', 'qty_pza', 'qty']);
                        if ($qtyCol) $upd2[$qtyCol] = (float)($sr['qty'] ?? 0);
                        if (table_has_column($pdo, 'orders', 'total')) $upd2['total'] = (float)($sr['total'] ?? 0);
                        if (table_has_column($pdo, 'orders', 'updated_at')) $upd2['updated_at'] = date('Y-m-d H:i:s');

                        if ($upd2) update_dynamic_by_id($pdo, 'orders', $orderId, $upd2);
                    }
                }

                $pdo->commit();
            } catch (Throwable $e) {
                if ($pdo->inTransaction()) $pdo->rollBack();
                throw $e;
            }


            // Push internos
            if (function_exists('sendPushToRole')) {
                $title = 'Pedido cancelado';
                $body  = 'Pedido #' . $orderId . ' fue cancelado por el cliente.';
                push_try(fn() => sendPushToRole($pdo, 'ventas', $title, $body, ['url' => './']), $DEBUG);
                push_try(fn() => sendPushToRole($pdo, 'admin',  $title, $body, ['url' => './']), $DEBUG);
                push_try(fn() => sendPushToRole($pdo, 'tienda', $title, $body, ['url' => './']), $DEBUG);
            }

            $txtItems = items_desc($items);
            $lineaHora  = !empty($pedido['delivery_time']) ? "Hora original: {$pedido['delivery_time']}\n" : '';

            $mensajeVentas = "Cancelación de pedido.\n\n"
                . "El cliente {$pedido['cliente']} ({$pedido['phone']}) canceló el pedido #{$pedido['id']}.\n"
                . "Entrega programada: {$pedido['delivery_date']}\n"
                . $lineaHora
                . "Dirección: {$pedido['delivery_address']}\n\n"
                . "Artículos:\n"
                . ($txtItems ?: "- Sin detalle de artículos\n");

            $wa_result = null;
            if ($WA_ENABLED && function_exists('wa_queue_to_role')) {
                $wa_result = wa_try(fn() => wa_queue_to_role($pdo, 'ventas', 'order_cancelled', (int)$orderId, $mensajeVentas), $DEBUG);
                wa_try_process($pdo, $DEBUG);
            }

            $sms_result = ['ventas' => null, 'cliente' => null];
            if ($SMS_ENABLED && function_exists('sms_queue_to_role')) {
                $smsVentas = "[Ventas Blanquita] Pedido CANCELADO #{$orderId}. Cliente: " . (string)($pedido['cliente'] ?? '') . ".";
                $sms_result['ventas'] = sms_try(fn() => sms_queue_to_role($pdo, 'ventas', 'order_cancelled', (int)$orderId, $smsVentas), $DEBUG);
                sms_try_process($pdo, $DEBUG);
            }

            if ($SMS_ENABLED && function_exists('sms_queue_to_phone')) {
                $countryHint = (string)($pedido['phone_country'] ?? 'MX');
                $smsCliente = "[Ventas Blanquita] Tu pedido #{$orderId} fue CANCELADO. Si fue un error, realiza un nuevo pedido desde la app.";
                $sms_result['cliente'] = sms_try(fn() => sms_queue_to_phone($pdo, (string)$pedido['phone'], 'order_cancelled_customer', (int)$orderId, $smsCliente, $countryHint), $DEBUG);
                sms_try_process($pdo, $DEBUG);
            }

            $waLink = null;
            $storePhone = trim((string)($CFG['store_phone'] ?? ''));
            if ($storePhone !== '') $waLink = wa_link_from_phone($storePhone, $mensajeVentas);

            out_json(['ok' => true, 'status' => 'cancelada', 'wa' => $wa_result, 'wa_link' => $waLink, 'sms' => $sms_result], $jsonOpts);
        }

        /* ================== CAMBIAR STATUS ================== */
        case 'cambiar_status': {
            require_roles(['ventas','admin','tienda','repartidor'], $role, $jsonOpts);
            if ($_SERVER['REQUEST_METHOD'] !== 'POST') out_json(['ok'=>false,'error'=>'METHOD_NOT_ALLOWED'], $jsonOpts, 405);

            $input   = read_input();
            $orderId = (int)($input['order_id'] ?? 0);
            $status  = trim((string)($input['status'] ?? ''));

            $valid = ['pendiente','autorizada','en_ruta','entregada','cancelada','rechazada'];
            if ($orderId <= 0 || !in_array($status, $valid, true)) out_json(['ok'=>false,'error'=>'INVALID_DATA'], $jsonOpts, 400);

            $upd = [];
            if (table_has_column($pdo, 'orders', 'status')) $upd['status'] = $status;
            if (table_has_column($pdo, 'orders', 'updated_at')) $upd['updated_at'] = date('Y-m-d H:i:s');
            update_dynamic_by_id($pdo, 'orders', $orderId, $upd);

            // PUSH al cliente
            if (function_exists('sendPushToUser')) {
                $ord = $pdo->prepare("SELECT id, customer_id FROM orders WHERE id = ? LIMIT 1");
                $ord->execute([$orderId]);
                $o = $ord->fetch(PDO::FETCH_ASSOC);
                if ($o && (int)$o['customer_id'] > 0) {
                    $title = 'Actualización de pedido';
                    $body  = 'Tu pedido #' . $orderId . ' ahora está: ' . $status;
                    push_try(fn() => sendPushToUser($pdo, (int)$o['customer_id'], $title, $body, ['url' => './']), $DEBUG);
                }
            }

            // WhatsApp/SMS al cliente (autorizada/cancelada/rechazada)
            $wa_result = null;
            $sms_result = null;

            if (in_array($status, ['autorizada','cancelada','rechazada'], true)) {
                $selTime  = table_has_column($pdo, 'orders', 'delivery_time') ? "o.delivery_time" : "NULL AS delivery_time";
                $selCountry = table_has_column($pdo, 'users', 'phone_country') ? "u.phone_country AS phone_country" : "'MX' AS phone_country";

                $stmt = $pdo->prepare("
                    SELECT o.id, o.delivery_date, {$selTime}, o.delivery_address,
                           u.name AS cliente, u.phone AS phone, {$selCountry}
                    FROM orders o
                    JOIN users u ON u.id = o.customer_id
                    WHERE o.id = ?
                    LIMIT 1
                ");
                $stmt->execute([$orderId]);
                $pedido = $stmt->fetch(PDO::FETCH_ASSOC);

                if ($pedido) {
                    $clienteNombre = (string)$pedido['cliente'];
                    $clienteTel    = (string)$pedido['phone'];
                    $countryHint   = (string)($pedido['phone_country'] ?? 'MX');
                    $lineaHora = !empty($pedido['delivery_time']) ? "Hora: {$pedido['delivery_time']}\n" : '';

                    if ($status === 'autorizada') {
                        $msg = "Hola {$clienteNombre}.\nTu pedido #{$pedido['id']} ha sido AUTORIZADO.\n\n";
                    } elseif ($status === 'cancelada') {
                        $msg = "Hola {$clienteNombre}.\nTu pedido #{$pedido['id']} ha sido CANCELADO.\n\n";
                    } else {
                        $msg = "Hola {$clienteNombre}.\nTu pedido #{$pedido['id']} ha sido RECHAZADO.\nSi necesitas información adicional, contáctanos.\n\n";
                    }

                    $msg .= "Fecha: {$pedido['delivery_date']}\n";
                    $msg .= $lineaHora;
                    $msg .= "Dirección: {$pedido['delivery_address']}\n";

                    if ($WA_ENABLED && function_exists('wa_queue_to_phone')) {
                        $wa_result = wa_try(fn() => wa_queue_to_phone($pdo, $clienteTel, 'order_status_'.$status, (int)$orderId, $msg, $countryHint), $DEBUG);
                        wa_try_process($pdo, $DEBUG);
                    }
                    if ($SMS_ENABLED && function_exists('sms_queue_to_phone')) {
                        $sms_result = sms_try(fn() => sms_queue_to_phone($pdo, $clienteTel, 'order_status_'.$status, (int)$orderId, $msg, $countryHint), $DEBUG);
                        sms_try_process($pdo, $DEBUG);
                    }
                }
            }

            out_json(['ok' => true, 'wa' => $wa_result, 'sms' => $sms_result], $jsonOpts);
        }


        /* ================== DETALLE PEDIDO (VENTAS) ================== */
        case 'pedido_detalle': {
            require_roles(['ventas','admin'], $role, $jsonOpts);

            $orderId = (int)($_GET['order_id'] ?? 0);
            if ($orderId <= 0) out_json(['ok'=>false,'error'=>'INVALID_ORDER'], $jsonOpts, 400);

            $selTime  = table_has_column($pdo, 'orders', 'delivery_time') ? "o.delivery_time" : "NULL AS delivery_time";
            $selSales = table_has_column($pdo, 'orders', 'sales_order') ? "o.sales_order" : "NULL AS sales_order";

            $stmt = $pdo->prepare("
                SELECT
                    o.id,
                    o.status,
                    o.delivery_date,
                    {$selTime},
                    o.delivery_address,
                    {$selSales},
                    u.name AS cliente,
                    u.phone AS phone,
                    o.customer_id
                FROM orders o
                JOIN users u ON u.id = o.customer_id
                WHERE o.id = ?
                LIMIT 1
            ");
            $stmt->execute([$orderId]);
            $order = $stmt->fetch(PDO::FETCH_ASSOC) ?: null;
            if (!$order) out_json(['ok'=>false,'error'=>'NOT_FOUND'], $jsonOpts, 404);

            $it = $pdo->prepare("
                SELECT oi.product_id, p.name, oi.quantity, oi.unit_price
                FROM order_items oi
                JOIN products p ON p.id = oi.product_id
                WHERE oi.order_id = ?
                ORDER BY p.name ASC
            ");
            $it->execute([$orderId]);
            $items = $it->fetchAll(PDO::FETCH_ASSOC) ?: [];

            out_json(['ok'=>true, 'order'=>$order, 'items'=>$items], $jsonOpts);
        }

        /* ================== LISTADO ADMIN ================== */
        case 'pedidos_admin': {
            require_roles(['ventas','admin','tienda','repartidor'], $role, $jsonOpts);

            $status = trim((string)($_GET['status'] ?? 'pendiente'));
            $valid = ['pendiente','autorizada','en_ruta','entregada','cancelada','rechazada'];
            if (!in_array($status, $valid, true)) out_json(['ok'=>false,'error'=>'INVALID_STATUS'], $jsonOpts, 400);

            $customerId = (int)($_GET['customer_id'] ?? 0);

            $select = [
                "o.id",
                "o.status",
                "o.delivery_date",
                "o.delivery_address",
                "u.name AS cliente",
                "u.phone AS phone",
                "o.customer_id",
                "SUM(oi.quantity) AS qty",
                "GROUP_CONCAT(CONCAT(p.name, ' x ', oi.quantity) SEPARATOR ', ') AS items_desc",
            ];
            if (table_has_column($pdo, 'orders', 'delivery_time')) $select[] = "o.delivery_time";
            if (table_has_column($pdo, 'orders', 'delivery_label')) $select[] = "o.delivery_label";
            if (table_has_column($pdo, 'orders', 'sales_order')) $select[] = "o.sales_order";

            $groupBy = [];
            foreach ($select as $s) {
                if (stripos($s, 'SUM(') !== false) continue;
                if (stripos($s, 'GROUP_CONCAT(') !== false) continue;
                $groupBy[] = preg_replace('/\s+AS\s+.*/i', '', $s);
            }

            $orderBy = "o.delivery_date ASC";
            $orderBy .= table_has_column($pdo, 'orders', 'created_at') ? ", o.created_at ASC" : ", o.id ASC";

            $where = "o.status = ?";
            $params = [$status];
            if ($customerId > 0) { $where .= " AND o.customer_id = ?"; $params[] = $customerId; }

            $sql = "
                SELECT
                    ".implode(",\n                    ", $select)."
                FROM orders o
                JOIN users u             ON u.id = o.customer_id
                LEFT JOIN order_items oi ON oi.order_id = o.id
                LEFT JOIN products p     ON p.id = oi.product_id
                WHERE {$where}
                GROUP BY ".implode(", ", $groupBy)."
                ORDER BY {$orderBy}
            ";

            $stmt = $pdo->prepare($sql);
            $stmt->execute($params);
            out_json(['ok' => true, 'rows' => $stmt->fetchAll(PDO::FETCH_ASSOC)], $jsonOpts);
        }

        /* ================== REGISTRAR OV (VENTAS) ================== */
        case 'registrar_ov': {
            require_roles(['ventas','admin'], $role, $jsonOpts);
            if ($_SERVER['REQUEST_METHOD'] !== 'POST') out_json(['ok'=>false,'error'=>'METHOD_NOT_ALLOWED'], $jsonOpts, 405);

            $input         = read_input();
            $orderId       = (int)($input['order_id'] ?? 0);
            $sales_order   = trim((string)($input['sales_order'] ?? ''));
            $status        = trim((string)($input['status'] ?? 'autorizada'));
            $delivery_time = trim((string)($input['delivery_time'] ?? ''));

            $delivery_date = trim((string)($input['delivery_date'] ?? ''));
            $itemsInput    = $input['items'] ?? null;

            if ($delivery_time !== '' && !preg_match('/^\d{2}:\d{2}$/', $delivery_time)) {
                out_json(['ok'=>false,'error'=>'DELIVERY_TIME_INVALID'], $jsonOpts, 400);
            }
            if ($delivery_date !== '' && !preg_match('/^\d{4}-\d{2}-\d{2}$/', $delivery_date)) {
                out_json(['ok'=>false,'error'=>'DELIVERY_DATE_INVALID'], $jsonOpts, 400);
            }

            $valid = ['pendiente','autorizada','en_ruta','entregada','cancelada','rechazada'];
            if ($orderId <= 0 || !in_array($status, $valid, true)) out_json(['ok'=>false,'error'=>'INVALID_DATA'], $jsonOpts, 400);

            $upd = [];
            if (table_has_column($pdo, 'orders', 'sales_order')) $upd['sales_order'] = ($sales_order !== '') ? $sales_order : null;
            if (table_has_column($pdo, 'orders', 'status')) $upd['status'] = $status;
            if (table_has_column($pdo, 'orders', 'sales_at')) $upd['sales_at'] = ($sales_order !== '') ? date('Y-m-d H:i:s') : null;
            if (table_has_column($pdo, 'orders', 'delivery_time')) $upd['delivery_time'] = ($delivery_time !== '') ? $delivery_time : null;
            if ($delivery_date !== '' && table_has_column($pdo, 'orders', 'delivery_date')) $upd['delivery_date'] = $delivery_date;
            if (table_has_column($pdo, 'orders', 'updated_at')) $upd['updated_at'] = date('Y-m-d H:i:s');

            $pdo->beginTransaction();
            try {
                update_dynamic_by_id($pdo, 'orders', $orderId, $upd);

                // Actualizar cantidades por artículo (opcional)
                if (is_array($itemsInput)) {
                    $cur = $pdo->prepare("SELECT product_id FROM order_items WHERE order_id = ?");
                    $cur->execute([$orderId]);
                    $existing = $cur->fetchAll(PDO::FETCH_ASSOC) ?: [];
                    $existingSet = [];
                    foreach ($existing as $r) { $existingSet[(int)$r['product_id']] = true; }

                    $clean = [];
                    foreach ($itemsInput as $it) {
                        $pid = (int)($it['product_id'] ?? 0);
                        $qty = (float)($it['qty'] ?? 0);
                        if ($pid > 0 && isset($existingSet[$pid])) {
                            $clean[$pid] = $qty;
                        }
                    }

                    if ($clean) {
                        $stmtUpd = $pdo->prepare("UPDATE order_items SET quantity = ? WHERE order_id = ? AND product_id = ?");
                        $stmtDel = $pdo->prepare("DELETE FROM order_items WHERE order_id = ? AND product_id = ?");

                        foreach ($clean as $pid => $qty) {
                            if ($qty <= 0) {
                                $stmtDel->execute([$orderId, $pid]);
                            } else {
                                $stmtUpd->execute([$qty, $orderId, $pid]);
                            }
                        }

                        $chk = $pdo->prepare("SELECT COUNT(*) FROM order_items WHERE order_id = ?");
                        $chk->execute([$orderId]);
                        $cnt = (int)$chk->fetchColumn();
                        if ($cnt <= 0) {
                            throw new RuntimeException('NO_ITEMS_LEFT');
                        }

                        $sum = $pdo->prepare("SELECT SUM(quantity) AS qty, SUM(quantity * unit_price) AS total FROM order_items WHERE order_id = ?");
                        $sum->execute([$orderId]);
                        $tot = $sum->fetch(PDO::FETCH_ASSOC) ?: ['qty'=>0,'total'=>0];

                        $upd2 = [];
                        $qtyCol = pick_column($pdo, 'orders', ['Qty_pza', 'qty_pza', 'qty']);
                        if ($qtyCol) $upd2[$qtyCol] = (float)($tot['qty'] ?? 0);
                        if (table_has_column($pdo, 'orders', 'total')) $upd2['total'] = (float)($tot['total'] ?? 0);

                        if ($upd2) update_dynamic_by_id($pdo, 'orders', $orderId, $upd2);
                    }
                }

                $pdo->commit();
            } catch (Throwable $e) {
                if ($pdo->inTransaction()) $pdo->rollBack();
                if ($e instanceof RuntimeException && $e->getMessage() === 'NO_ITEMS_LEFT') {
                    out_json(['ok'=>false,'error'=>'ITEMS_REQUIRED'], $jsonOpts, 400);
                }
                throw $e;
            }
            // PUSH
            if (function_exists('sendPushToUser') || function_exists('sendPushToRole')) {
                $tmp = $pdo->prepare("SELECT id, customer_id FROM orders WHERE id = ? LIMIT 1");
                $tmp->execute([$orderId]);
                $ord = $tmp->fetch(PDO::FETCH_ASSOC) ?: null;

                $title = 'Actualización de pedido';
                $body  = 'Pedido #' . $orderId . ' · Nuevo estatus: ' . $status;
                if ($sales_order !== '') $body .= ' · OC: ' . $sales_order;

                push_try(function() use ($pdo, $ord, $title, $body) {
                    if ($ord && function_exists('sendPushToUser')) sendPushToUser($pdo, (int)$ord['customer_id'], $title, $body, ['url' => './']);
                    if (function_exists('sendPushToRole')) sendPushToRole($pdo, 'tienda', $title, $body, ['url' => './']);
                }, $DEBUG);
            }

            // Notificar al cliente si estatus relevante
            $wa_result = null;
            $waLink = null;

            if (in_array($status, ['autorizada','cancelada','rechazada'], true)) {
                $selSales = table_has_column($pdo, 'orders', 'sales_order') ? "o.sales_order" : "NULL AS sales_order";
                $selTime  = table_has_column($pdo, 'orders', 'delivery_time') ? "o.delivery_time" : "NULL AS delivery_time";
                $selCountry = table_has_column($pdo, 'users', 'phone_country') ? "u.phone_country AS phone_country" : "'MX' AS phone_country";

                $stmt = $pdo->prepare("
                    SELECT o.id, o.delivery_date, {$selTime}, o.delivery_address, o.status, {$selSales},
                           u.name AS cliente, u.phone AS phone, {$selCountry}
                    FROM orders o
                    JOIN users u ON u.id = o.customer_id
                    WHERE o.id = ?
                    LIMIT 1
                ");
                $stmt->execute([$orderId]);
                $pedido = $stmt->fetch(PDO::FETCH_ASSOC);

                if ($pedido) {
                    $clienteNombre = (string)$pedido['cliente'];
                    $clienteTel    = (string)$pedido['phone'];
                    $countryHint   = (string)($pedido['phone_country'] ?? 'MX');
                    $ocTxt         = !empty($pedido['sales_order']) ? "OC: {$pedido['sales_order']}\n" : '';
                    $lineaHora     = !empty($pedido['delivery_time']) ? "Hora de entrega: {$pedido['delivery_time']}\n" : '';

                    if ($status === 'autorizada') {
                        $mensaje = "Hola {$clienteNombre}.\nTu pedido #{$pedido['id']} ha sido AUTORIZADO.\n\n"
                            . $ocTxt
                            . "Fecha de entrega: {$pedido['delivery_date']}\n"
                            . $lineaHora
                            . "Dirección: {$pedido['delivery_address']}\n";
                    } elseif ($status === 'cancelada') {
                        $mensaje = "Hola {$clienteNombre}.\nTu pedido #{$pedido['id']} ha sido CANCELADO.\n\n"
                            . $ocTxt
                            . "Fecha solicitada: {$pedido['delivery_date']}\n"
                            . $lineaHora
                            . "Dirección: {$pedido['delivery_address']}\n";
                    } else {
                        $mensaje = "Hola {$clienteNombre}.\nTu pedido #{$pedido['id']} ha sido RECHAZADO.\n"
                            . "Si necesitas información adicional, por favor contáctanos.\n\n"
                            . "Fecha solicitada: {$pedido['delivery_date']}\n"
                            . $lineaHora
                            . "Dirección: {$pedido['delivery_address']}\n";
                    }

                    if ($WA_ENABLED && function_exists('wa_queue_to_phone')) {
                        $wa_result = wa_try(fn() => wa_queue_to_phone($pdo, $clienteTel, 'order_status_'.$status, (int)$orderId, $mensaje, $countryHint), $DEBUG);
                        wa_try_process($pdo, $DEBUG);
                    }
                    $waLink = wa_link_from_phone($clienteTel, $mensaje);
                }
            }

            out_json(['ok' => true, 'status' => $status, 'wa' => $wa_result, 'wa_link' => $waLink], $jsonOpts);
        }

        /* ================== HISTORIAL (VENTAS) ================== */
        case 'historial_pedidos': {
            require_roles(['ventas','admin'], $role, $jsonOpts);

            $desde = trim((string)($_GET['desde'] ?? ''));
            $hasta = trim((string)($_GET['hasta'] ?? ''));
            $customerId = (int)($_GET['customer_id'] ?? 0);

            if ($desde === '' || $hasta === '') out_json(['ok'=>false,'error'=>'DATE_RANGE_REQUIRED'], $jsonOpts, 400);

            $desdeFull = $desde . ' 00:00:00';
            $hastaFull = $hasta . ' 23:59:59';

            if (!table_has_column($pdo, 'orders', 'created_at')) out_json(['ok'=>false,'error'=>'CREATED_AT_REQUIRED_IN_SCHEMA'], $jsonOpts, 400);

            $selSales = table_has_column($pdo, 'orders', 'sales_order') ? "o.sales_order" : "NULL AS sales_order";

            $where = "o.created_at BETWEEN ? AND ?";
            $params = [$desdeFull, $hastaFull];
            if ($customerId > 0) { $where .= " AND o.customer_id = ?"; $params[] = $customerId; }

            $stmt = $pdo->prepare("
                SELECT
                    o.id,
                    o.created_at,
                    {$selSales} AS sales_order,
                    o.delivery_date,
                    o.delivery_address,
                    o.status,
                    u.name  AS cliente,
                    u.phone AS phone,
                    o.customer_id,
                    SUM(oi.quantity) AS qty
                FROM orders o
                JOIN users u             ON u.id = o.customer_id
                LEFT JOIN order_items oi ON oi.order_id = o.id
                WHERE {$where}
                GROUP BY
                    o.id, o.created_at, o.delivery_date, o.delivery_address, o.status,
                    u.name, u.phone, o.customer_id, {$selSales}
                ORDER BY o.created_at DESC
            ");
            $stmt->execute($params);
            out_json(['ok' => true, 'rows' => $stmt->fetchAll(PDO::FETCH_ASSOC)], $jsonOpts);
        }

        /* ================== VENTAS: CLIENTES (para filtros) ================== */
        case 'ventas_clientes': {
            require_roles(['ventas','admin'], $role, $jsonOpts);

            $stmt = $pdo->query("
                SELECT id, name
                FROM users
                WHERE role = 'cliente' AND is_active = 1
                ORDER BY name ASC, id ASC
            ");
            out_json(['ok' => true, 'rows' => ($stmt->fetchAll(PDO::FETCH_ASSOC) ?: [])], $jsonOpts);
        }

        /* ================== VENTAS: DASHBOARD ================== */
        case 'ventas_dashboard': {
            require_roles(['ventas','admin'], $role, $jsonOpts);

            $desde = trim((string)($_GET['desde'] ?? ''));
            $hasta = trim((string)($_GET['hasta'] ?? ''));
            $customerId = (int)($_GET['customer_id'] ?? 0);

            // defaults: últimos 30 días si no se manda
            if ($hasta === '') $hasta = date('Y-m-d');
            if ($desde === '') $desde = date('Y-m-d', strtotime('-30 days'));

            $desdeFull = $desde . ' 00:00:00';
            $hastaFull = $hasta . ' 23:59:59';

            if (!table_has_column($pdo, 'orders', 'created_at')) {
                out_json(['ok'=>false,'error'=>'CREATED_AT_REQUIRED_IN_SCHEMA'], $jsonOpts, 400);
            }

            $where = "o.created_at BETWEEN ? AND ?";
            $params = [$desdeFull, $hastaFull];
            if ($customerId > 0) { $where .= " AND o.customer_id = ?"; $params[] = $customerId; }

            // KPIs por estatus (en una sola consulta)
            $kpiSql = "
                SELECT
                    SUM(CASE WHEN o.status='pendiente'  THEN 1 ELSE 0 END) AS pendientes,
                    SUM(CASE WHEN o.status='autorizada' THEN 1 ELSE 0 END) AS autorizados,
                    SUM(CASE WHEN o.status='en_ruta'    THEN 1 ELSE 0 END) AS en_ruta,
                    SUM(CASE WHEN o.status='entregada'  THEN 1 ELSE 0 END) AS entregados,
                    SUM(CASE WHEN o.status='cancelada'  THEN 1 ELSE 0 END) AS cancelados,
                    SUM(CASE WHEN o.status='rechazada'  THEN 1 ELSE 0 END) AS rechazados,
                    COUNT(*) AS total
                FROM orders o
                WHERE {$where}
            ";
            $st = $pdo->prepare($kpiSql);
            $st->execute($params);
            $kpis = $st->fetch(PDO::FETCH_ASSOC) ?: [];

            // Top productos
            $topSql = "
                SELECT p.id, p.name, SUM(oi.quantity) AS qty
                FROM orders o
                JOIN order_items oi ON oi.order_id = o.id
                JOIN products p     ON p.id = oi.product_id
                WHERE {$where}
                GROUP BY p.id, p.name
                ORDER BY qty DESC
                LIMIT 8
            ";
            $st2 = $pdo->prepare($topSql);
            $st2->execute($params);
            $top = $st2->fetchAll(PDO::FETCH_ASSOC) ?: [];

            out_json(['ok' => true, 'kpis' => $kpis, 'top_products' => $top], $jsonOpts);
        }

        default:
            out_json(['ok' => false, 'error' => 'UNKNOWN_ACTION'], $jsonOpts, 400);
    }

} catch (Throwable $e) {
    $out = ['ok' => false, 'error' => 'EXCEPTION'];
    if ($DEBUG) $out['detail'] = $e->getMessage();
    out_json($out, $jsonOpts, 500);
}
