// app.js - Ventas Blanquita (con pestañas Ventas + KPIs)

// Toast bootstrap
function toast(msg) {
    const body = document.getElementById('toastBody');
    const toastEl = document.getElementById('toastMsg');
    if (!body || !toastEl) return;
    body.textContent = String(msg || '');
    const t = new bootstrap.Toast(toastEl);
    t.show();
}

// Escape HTML para render seguro
function escapeHTML(s) {
    return String(s ?? '')
        .replace(/&/g, '&amp;')
        .replace(/</g, '&lt;')
        .replace(/>/g, '&gt;')
        .replace(/"/g, '&quot;')
        .replace(/'/g, '&#039;');
}

function getVal(selector) {
    const el = document.querySelector(selector);
    return el ? el.value : '';
}

function urlBase64ToUint8Array(base64String) {
    const padding = '='.repeat((4 - base64String.length % 4) % 4);
    const base64 = (base64String + padding).replace(/-/g, '+').replace(/_/g, '/');
    const rawData = atob(base64);
    const outputArray = new Uint8Array(rawData.length);
    for (let i = 0; i < rawData.length; ++i) outputArray[i] = rawData.charCodeAt(i);
    return outputArray;
}

// Service Worker
if ('serviceWorker' in navigator) {
    navigator.serviceWorker.register('service-worker.js').catch(console.error);
}

// PWA Install
let deferredInstallPrompt = null;
window.addEventListener('beforeinstallprompt', (e) => {
    e.preventDefault();
    deferredInstallPrompt = e;
    const btn = document.getElementById('btnInstallApp');
    if (btn) btn.classList.remove('d-none');
});

// API helper (sin caché)
async function api(action, data = null, method = 'POST') {
    method = (method || 'POST').toUpperCase();
    let url = 'api.php?action=' + encodeURIComponent(action);

    if (method === 'GET') {
        const qs = new URLSearchParams();
        if (data && typeof data === 'object') {
            for (const [k, v] of Object.entries(data)) {
                if (v === undefined || v === null) continue;
                qs.append(k, String(v));
            }
        }
        qs.append('_ts', String(Date.now()));
        url += '&' + qs.toString();
    }

    const opt = { method, cache: 'no-store', headers: {} };

    if (method !== 'GET' && data !== null) {
        opt.headers['Content-Type'] = 'application/json; charset=utf-8';
        opt.body = JSON.stringify(data);
    }

    const res = await fetch(url, opt);
    const txt = await res.text();

    let json;
    try {
        json = JSON.parse(txt);
    } catch (e) {
        console.error('Respuesta no JSON desde', url, txt.slice(0, 800));
        throw new Error('EXCEPTION');
    }

    if (!res.ok) {
        console.error('HTTP error', res.status, json);
        throw new Error((json && (json.error || json.message)) ? (json.error || json.message) : 'EXCEPTION');
    }
    return json;
}

document.addEventListener('DOMContentLoaded', () => {
    const role = window.CURRENT_ROLE;

    // Push enable
    const btnEnablePush = document.getElementById('btnEnablePush');

    async function subscribePush() {
        if (!('Notification' in window) || !('serviceWorker' in navigator)) {
            toast('Tu navegador no soporta notificaciones push.');
            return;
        }

        const perm = await Notification.requestPermission();
        if (perm !== 'granted') {
            toast('No se activaron las notificaciones (permiso denegado).');
            return;
        }

        const reg = await navigator.serviceWorker.ready;

        let vapidPublic = '';
        try {
            const k = await api('push_public_key', null, 'GET');
            vapidPublic = (k.publicKey || '').trim();
        } catch (e) {
            console.warn('No se pudo leer push_public_key', e);
        }

        if (!vapidPublic) {
            toast('Notificaciones aún no están configuradas en el servidor.');
            return;
        }

        let sub = await reg.pushManager.getSubscription();
        if (!sub) {
            sub = await reg.pushManager.subscribe({
                userVisibleOnly: true,
                applicationServerKey: urlBase64ToUint8Array(vapidPublic)
            });
        }

        const payload = (typeof sub.toJSON === 'function') ? sub.toJSON() : sub;

        try {
            const r = await api('push_subscribe', payload, 'POST');
            if (!r.ok) throw new Error(r.error || 'Error al registrar suscripción');
            toast('Notificaciones activadas para tu usuario.');
        } catch (e) {
            console.error('Error en push_subscribe', e);
            toast(e.message);
        }
    }

    if (btnEnablePush && 'Notification' in window && 'serviceWorker' in navigator) {
        btnEnablePush.classList.remove('d-none');
        btnEnablePush.addEventListener('click', () => subscribePush().catch(console.error));
    }

    // Install button
    const btnInstall = document.getElementById('btnInstallApp');
    if (btnInstall) {
        btnInstall.addEventListener('click', async () => {
            if (!deferredInstallPrompt) {
                toast('La instalación no está disponible en este momento.');
                return;
            }
            deferredInstallPrompt.prompt();
            try {
                await deferredInstallPrompt.userChoice;
            } catch (err) {
                console.error('Error en userChoice PWA', err);
            }
            deferredInstallPrompt = null;
            btnInstall.classList.add('d-none');
        });
    }

    // QR modal
    const modalQREl = document.getElementById('modalQR');
    const qrCanvas = document.getElementById('qrCanvas');
    const qrOrderInfo = document.getElementById('qrOrderInfo');
    const modalQR = modalQREl ? new bootstrap.Modal(modalQREl) : null;

    function mostrarQR(order) {
        if (!modalQR || !qrCanvas || typeof QRCode === 'undefined') return;
        const payload = `ENTREGA|${order.id}|${order.qty}`;
        const ctx = qrCanvas.getContext('2d');
        if (ctx) ctx.clearRect(0, 0, qrCanvas.width, qrCanvas.height);

        QRCode.toCanvas(qrCanvas, payload, { width: 220 }, (err) => {
            if (err) console.error(err);
        });

        if (qrOrderInfo) {
            qrOrderInfo.textContent = `Pedido #${order.id} · Piezas: ${order.qty} · Entrega ${order.delivery_date}`;
        }
        modalQR.show();
    }

    /* ================== CLIENTE ================== */
    if (role === 'cliente') {
        const btnCrear = document.getElementById('btnCrearPedido');
        const frm = document.getElementById('frmPedido');
        const msg = document.getElementById('msgPedido');

        // ================== Reglas de entrega (Cliente) ==================
        // Requerimiento:
        // 1) Si el cliente crea un pedido a las 08:00, la entrega mínima debe iniciar a las 14:00 (NOW + 6h).
        // 2) Si el pedido se inicia después de las 18:00, ya no se permite seleccionar el mismo día.
        //    La entrega será a partir del día siguiente y la hora mínima será 06:00.
        const inpDeliveryDate = frm ? frm.querySelector('[name="delivery_date"]') : null;
        const inpDeliveryTime = frm ? frm.querySelector('[name="delivery_time"]') : null;

        function pad2(n) {
            return String(n).padStart(2, '0');
        }

        function toYMD(d) {
            const y = d.getFullYear();
            const m = pad2(d.getMonth() + 1);
            const day = pad2(d.getDate());
            return `${y}-${m}-${day}`;
        }

        function toHM(d) {
            return `${pad2(d.getHours())}:${pad2(d.getMinutes())}`;
        }

        function addDays(d, n) {
            const x = new Date(d.getTime());
            x.setDate(x.getDate() + n);
            return x;
        }

        function applyDeliveryRules(showToastOnAdjust = false) {
            if (!inpDeliveryDate) return;

            const now = new Date();
            const cutoff = new Date(now.getTime());
            cutoff.setHours(18, 0, 0, 0);

            const todayYMD = toYMD(now);
            const minDateObj = (now >= cutoff) ? addDays(now, 1) : now;
            const minDateYMD = toYMD(minDateObj);

            // Si ya pasó de las 18:00, bloquea el mismo día
            inpDeliveryDate.min = minDateYMD;
            if (!inpDeliveryDate.value || inpDeliveryDate.value < inpDeliveryDate.min) {
                inpDeliveryDate.value = inpDeliveryDate.min;
                if (showToastOnAdjust) toast('Después de las 6:00 pm, la entrega solo puede ser a partir del día siguiente.');
            }

            // Hora mínima
            let minTime = '06:00';
            if (inpDeliveryDate.value === todayYMD && now < cutoff) {
                const minDT = new Date(now.getTime() + (6 * 60 * 60 * 1000));
                minTime = toHM(minDT);
            }

            if (inpDeliveryTime) {
                inpDeliveryTime.min = minTime;
                if (!inpDeliveryTime.value || inpDeliveryTime.value < minTime) {
                    inpDeliveryTime.value = minTime;
                    if (showToastOnAdjust) toast(`La hora mínima de entrega es ${minTime}.`);
                }
            }
        }

        // Inicializa y asegura que el usuario no pueda seleccionar una fecha/hora fuera de regla.
        applyDeliveryRules(false);
        inpDeliveryDate?.addEventListener('change', () => applyDeliveryRules(true));
        inpDeliveryTime?.addEventListener('change', () => applyDeliveryRules(true));

        const itemsContainer = document.getElementById('itemsContainer');
        const btnAddItem = document.getElementById('btnAddItem');

        const modalConfirmEl = document.getElementById('modalConfirmPedido');
        const modalCancelarEl = document.getElementById('modalCancelarPedido');
        const confirmResumenEl = document.getElementById('confirmPedidoResumen');
        const cancelInfoEl = document.getElementById('cancelPedidoInfo');
        const btnConfirmEnviar = document.getElementById('btnConfirmarEnviar');
        const btnConfirmCanc = document.getElementById('btnConfirmarCancelacion');

        const modalConfirm = modalConfirmEl ? new bootstrap.Modal(modalConfirmEl) : null;
        const modalCancelar = modalCancelarEl ? new bootstrap.Modal(modalCancelarEl) : null;

        let pedidoPorConfirmar = null;
        let pedidoPorCancelar = null;

        function refreshRemoveButtons() {
            const rows = itemsContainer?.querySelectorAll('.order-item-row') || [];
            rows.forEach((row) => {
                const btn = row.querySelector('.btn-remove-item');
                if (btn) btn.disabled = (rows.length === 1);
            });
        }

        function keepSingleItemRow() {
            if (!itemsContainer) return;
            const rows = itemsContainer.querySelectorAll('.order-item-row');
            if (rows.length <= 1) return;
            for (let i = rows.length - 1; i >= 1; i--) rows[i].remove();
            refreshRemoveButtons();
        }

        btnAddItem?.addEventListener('click', () => {
            if (!itemsContainer) return;
            const rows = itemsContainer.querySelectorAll('.order-item-row');
            if (!rows.length) return;

            const base = rows[rows.length - 1];
            const clone = base.cloneNode(true);

            const qtyInput = clone.querySelector('.item-qty');
            if (qtyInput) qtyInput.value = '1';

            const prodSel = clone.querySelector('.item-product');
            if (prodSel) prodSel.value = '';

            itemsContainer.appendChild(clone);
            refreshRemoveButtons();
        });

        itemsContainer?.addEventListener('click', (ev) => {
            const btn = ev.target.closest('.btn-remove-item');
            if (!btn || !itemsContainer) return;
            const rows = itemsContainer.querySelectorAll('.order-item-row');
            if (rows.length <= 1) {
                toast('Debe haber al menos un artículo en el pedido.');
                return;
            }
            btn.closest('.order-item-row')?.remove();
            refreshRemoveButtons();
        });

        refreshRemoveButtons();

        // Direcciones
        const selAddress = document.getElementById('selAddress');
        const txtAddress = document.getElementById('txtDeliveryAddress');
        const btnNuevaDir = document.getElementById('btnNuevaDir');

        const modalDirEl = document.getElementById('modalDireccion');
        const modalDir = modalDirEl ? new bootstrap.Modal(modalDirEl) : null;

        const dirLabel = document.getElementById('dirLabel');
        const dirAddress = document.getElementById('dirAddress');
        const dirCP = document.getElementById('dirCP');

        const loadDirecciones = async () => {
            if (!selAddress || !txtAddress) return;

            try {
                const r = await api('direcciones_listar', null, 'GET');
                if (!r.ok) throw new Error(r.error || 'Error al cargar direcciones');

                const rows = r.rows || [];
                selAddress.innerHTML = '';

                if (!rows.length) {
                    const opt = document.createElement('option');
                    opt.value = txtAddress.value || '';
                    opt.textContent = txtAddress.value ? 'Dirección principal' : 'Sin direcciones guardadas';
                    selAddress.appendChild(opt);
                    return;
                }

                rows.forEach((d) => {
                    const opt = document.createElement('option');
                    opt.value = d.address;
                    opt.dataset.id = d.id;
                    opt.dataset.cp = d.postal_code || '';
                    opt.dataset.label = d.label || '';
                    opt.textContent = d.label
                        ? d.label
                        : (d.address.length > 50 ? d.address.substring(0, 50) + '…' : d.address);
                    selAddress.appendChild(opt);
                });

                txtAddress.value = rows[0].address;
            } catch (e) {
                console.error('Error en loadDirecciones', e);
                toast('No se pudieron cargar tus direcciones');
            }
        };

        selAddress?.addEventListener('change', () => {
            if (!selAddress || !txtAddress) return;
            txtAddress.value = selAddress.value || '';
        });

        btnNuevaDir?.addEventListener('click', () => {
            if (!modalDir) return;
            if (dirLabel) dirLabel.value = '';
            if (dirAddress) dirAddress.value = '';
            if (dirCP) dirCP.value = '';
            modalDir.show();
        });

        const btnGuardarDir = document.getElementById('btnGuardarDir');
        btnGuardarDir?.addEventListener('click', async () => {
            const data = {
                label: (dirLabel?.value || '').trim(),
                address: (dirAddress?.value || '').trim(),
                postal_code: (dirCP?.value || '').trim()
            };

            if (!data.address) {
                toast('La dirección es obligatoria');
                return;
            }

            try {
                const r = await api('direccion_crear', data, 'POST');
                if (!r.ok) throw new Error(r.error || 'No se pudo guardar la dirección');

                toast('Dirección guardada');
                modalDir && modalDir.hide();
                await loadDirecciones();
            } catch (e) {
                console.error('Error guardando dirección', e);
                toast(e.message);
            }
        });

        const loadPedidos = async () => {
            try {
                const r = await api('mis_pedidos', null, 'GET');
                const cont = document.getElementById('listaPedidos');
                if (!cont) return;

                if (!r.ok) throw new Error(r.error || 'Error');

                const rows = r.rows || [];
                if (!rows.length) {
                    cont.innerHTML = '<p class="text-muted">Sin pedidos aún.</p>';
                    return;
                }

                let html = '<div class="list-group">';
                rows.forEach(p => {
                    html += `
            <div class="list-group-item d-flex justify-content-between align-items-center">
              <div>
                <div><strong>Pedido #${escapeHTML(p.id)}</strong> — ${escapeHTML(p.status)}</div>
                <div class="small">Entrega: ${escapeHTML(p.delivery_date)}</div>
                <div class="small">Piezas: <strong>${escapeHTML(p.qty)}</strong></div>
              </div>
              <div class="text-end">
                <button type="button"
                        class="btn btn-sm btn-outline-primary btn-ver-qr"
                        data-id="${escapeHTML(p.id)}"
                        data-status="${escapeHTML(p.status)}"
                        data-delivery="${escapeHTML(p.delivery_date)}"
                        data-qty="${escapeHTML(p.qty)}">
                  Ver QR
                </button>
                ${p.status === 'pendiente'
                            ? `<button type="button"
                             class="btn btn-sm btn-outline-danger ms-2 btn-cancel-pedido"
                             data-id="${escapeHTML(p.id)}"
                             data-delivery="${escapeHTML(p.delivery_date)}"
                             data-qty="${escapeHTML(p.qty)}">
                       Cancelar
                     </button>`
                            : ''}
              </div>
            </div>`;
                });
                html += '</div>';
                cont.innerHTML = html;

                cont.onclick = (ev) => {
                    const btnCancel = ev.target.closest('.btn-cancel-pedido');
                    if (btnCancel) {
                        const id = btnCancel.dataset.id;
                        const qty = btnCancel.dataset.qty;
                        const fecha = btnCancel.dataset.delivery;
                        pedidoPorCancelar = { id, qty, fecha };

                        if (cancelInfoEl) {
                            cancelInfoEl.innerHTML = `
                <div>Pedido <strong>#${escapeHTML(id)}</strong></div>
                <div>Entrega: ${escapeHTML(fecha)}</div>
                <div>Piezas: ${escapeHTML(qty)}</div>
              `;
                        }
                        modalCancelar && modalCancelar.show();
                        return;
                    }

                    const btnQR = ev.target.closest('.btn-ver-qr');
                    if (!btnQR) return;

                    if (btnQR.dataset.status !== 'pendiente') {
                        toast('Solo los pedidos pendientes generan código de entrega.');
                        return;
                    }

                    mostrarQR({
                        id: btnQR.dataset.id,
                        qty: btnQR.dataset.qty,
                        delivery_date: btnQR.dataset.delivery
                    });
                };

            } catch (e) {
                console.error('Error en loadPedidos', e);
            }
        };

        // Crear -> abre modal confirmación
        btnCrear?.addEventListener('click', async () => {
            const items = [];
            const itemsResumen = [];

            if (itemsContainer) {
                itemsContainer.querySelectorAll('.order-item-row').forEach(row => {
                    const prodSel = row.querySelector('.item-product');
                    const qtyInp = row.querySelector('.item-qty');

                    const product_id = prodSel && prodSel.value ? parseInt(prodSel.value, 10) : 0;
                    const qty = qtyInp && qtyInp.value ? parseFloat(qtyInp.value) : 0;
                    const name = prodSel && prodSel.selectedOptions.length ? prodSel.selectedOptions[0].text.trim() : '';

                    if (product_id > 0 && qty > 0) {
                        items.push({ product_id, qty });
                        itemsResumen.push({ name, qty });
                    }
                });
            }

            if (!items.length) {
                toast('Agrega al menos un artículo con cantidad mayor a 0.');
                return;
            }

            const delivery_date = getVal('[name="delivery_date"]');
            const delivery_time = getVal('[name="delivery_time"]');
            const delivery_address = getVal('[name="delivery_address"]');
            const instructions = getVal('[name="instructions"]');
            const comments = getVal('[name="comments"]');

            if (!delivery_date) {
                toast('La fecha de entrega es obligatoria.');
                return;
            }

            let delivery_label = '';
            if (selAddress && selAddress.options.length > 0 && selAddress.selectedIndex >= 0) {
                const optSel = selAddress.options[selAddress.selectedIndex];
                if (optSel && optSel.dataset.label) delivery_label = optSel.dataset.label;
            }

            pedidoPorConfirmar = {
                data: { items, delivery_date, delivery_time, delivery_address, delivery_label, instructions, comments },
                resumen: { itemsResumen, delivery_date, delivery_time, delivery_address }
            };

            if (confirmResumenEl) {
                let html = '<ul class="mb-2">';
                pedidoPorConfirmar.resumen.itemsResumen.forEach(it => {
                    html += `<li>${escapeHTML(it.name || 'Artículo')} — <strong>${escapeHTML(it.qty)}</strong></li>`;
                });
                html += '</ul>';

                html += `<p class="mb-1"><strong>Fecha de entrega:</strong> ${escapeHTML(delivery_date)}</p>`;
                if (delivery_time) html += `<p class="mb-1"><strong>Hora aproximada:</strong> ${escapeHTML(delivery_time)}</p>`;
                if (delivery_address) html += `<p class="mb-0"><strong>Dirección de entrega:</strong> ${escapeHTML(delivery_address)}</p>`;
                confirmResumenEl.innerHTML = html;
            }

            modalConfirm && modalConfirm.show();
        });

        // Confirmar y enviar
        btnConfirmEnviar?.addEventListener('click', async () => {
            if (!pedidoPorConfirmar) {
                toast('No hay pedido por confirmar');
                return;
            }

            if (msg) msg.textContent = 'Enviando...';

            try {
                const r = await api('crear_pedido', pedidoPorConfirmar.data, 'POST');
                if (!r.ok) throw new Error(r.error || 'Error');

                if (msg) msg.textContent = `Pedido creado (#${r.order_id}).`;
                toast('Pedido enviado');

                modalConfirm && modalConfirm.hide();
                pedidoPorConfirmar = null;

                if (frm) frm.reset();
                keepSingleItemRow();
                applyDeliveryRules(false);
                await loadDirecciones();
                await loadPedidos();
            } catch (e) {
                console.error('Error creando pedido', e);
                if (msg) msg.textContent = e.message;
                else toast(e.message);
            }
        });

        // Confirmar cancelación
        btnConfirmCanc?.addEventListener('click', async () => {
            if (!pedidoPorCancelar) {
                toast('No hay pedido para cancelar');
                return;
            }

            try {
                const r = await api('cancelar_pedido', { order_id: Number(pedidoPorCancelar.id) }, 'POST');
                if (!r.ok) throw new Error(r.error || 'Error al cancelar');

                toast('Pedido cancelado');
                if (msg) msg.textContent = `Pedido #${pedidoPorCancelar.id} cancelado.`;

                modalCancelar && modalCancelar.hide();
                pedidoPorCancelar = null;

                await loadPedidos();
            } catch (e) {
                console.error('Error cancelando pedido', e);
                toast(e.message);
            }
        });

        loadDirecciones();
        loadPedidos();
    }

    /* ================== VENTAS ================== */
    if (role === 'ventas') {
        const cont = document.getElementById('listaPedidosVentas');

        const selClientePend = document.getElementById('ventasFiltroClientePend');
        const btnRefPend = document.getElementById('btnVentasRefrescarPend');

        const histDesde = document.getElementById('histDesde');
        const histHasta = document.getElementById('histHasta');
        const selClienteHist = document.getElementById('ventasFiltroClienteHist');
        const btnVerHist = document.getElementById('btnVerHistorial');
        const btnExpHist = document.getElementById('btnExportarHistorial');
        const histRes = document.getElementById('historialResultados');

        // KPI
        const kpiMes = document.getElementById('ventasKpiMes');
        const kpiCliente = document.getElementById('ventasKpiCliente');
        const btnRefKpi = document.getElementById('btnVentasRefrescarKpi');

        const kpiPend = document.getElementById('kpiPendientes');
        const kpiAut = document.getElementById('kpiAutorizados');
        const kpiEnt = document.getElementById('kpiEntregados');
        const kpiRech = document.getElementById('kpiRechazados');
        const kpiRechDet = document.getElementById('kpiRechazoDetalle');

        const chartCanvas = document.getElementById('chartTopProducts');
        const chartNote = document.getElementById('chartTopProductsNote');
        let chartTop = null;

        const modalVentasEl = document.getElementById('modalPedidoVentas');
        const modalVentas = modalVentasEl ? new bootstrap.Modal(modalVentasEl) : null;

        const infoPedido = document.getElementById('ventasInfoPedido');
        const inputOC = document.getElementById('ventasOc');
        
        const inputHora = document.getElementById('ventasHora');
        const selectStatus = document.getElementById('ventasStatus');
        const inputFecha = document.getElementById('ventasFecha');
        const itemsEditor = document.getElementById('ventasItemsEditor');
        const btnGuardarOC = document.getElementById('btnGuardarOC');
        const btnDescPDF = document.getElementById('btnDescargarPDF');

        let pedidoActual = null;

        function renderVentasItemsEditor(items) {
            if (!itemsEditor) return;
            if (!Array.isArray(items) || !items.length) {
                itemsEditor.innerHTML = '<div class="text-muted small">Sin artículos para editar.</div>';
                return;
            }

            itemsEditor.innerHTML = '';

            (items || []).forEach((it) => {
                const pid = Number(it.product_id || 0);
                const name = String(it.name || 'Producto');
                const qtyRaw = (it.quantity ?? it.qty ?? 0);
                const qty = Number(qtyRaw);

                const row = document.createElement('div');
                row.className = 'd-flex align-items-center gap-2';
                row.dataset.productId = String(pid);

                const safeName = escapeHTML(name);
                const safeQty = escapeHTML(String(isFinite(qty) ? qty : 0));

                row.innerHTML = `
                    <div class="flex-grow-1 small">${safeName}</div>
                    <input type="number" class="form-control form-control-sm" style="max-width:120px" min="0" step="1" value="${safeQty}">
                `;
                itemsEditor.appendChild(row);
            });
        }

        async function loadPedidoDetalleVentas(orderId) {
            if (!orderId) return;
            if (itemsEditor) itemsEditor.innerHTML = '<div class="text-muted small">Cargando artículos...</div>';

            try {
                const r = await api('pedido_detalle', { order_id: Number(orderId) }, 'GET');
                if (!r.ok) throw new Error(r.error || 'Error al cargar detalle');

                const ord = r.order || {};
                if (inputFecha && ord.delivery_date) inputFecha.value = ord.delivery_date;
                if (inputHora && ord.delivery_time) inputHora.value = ord.delivery_time || '';

                renderVentasItemsEditor(r.items || []);
            } catch (e) {
                console.error('Error loadPedidoDetalleVentas', e);
                if (itemsEditor) itemsEditor.innerHTML = '<div class="text-muted small">No se pudo cargar el detalle para editar artículos.</div>';
            }
        }

        async function loadClientesForFilters() {
            // Poblar selects (pendientes/historial/kpis)
            try {
                const r = await api('ventas_clientes', null, 'GET');
                if (!r.ok) throw new Error(r.error || 'Error');

                const rows = r.rows || [];

                function fillSel(sel) {
                    if (!sel) return;
                    const cur = sel.value || '';
                    const baseOptions = sel.querySelectorAll('option');
                    // conserva el primero (Todos)
                    const first = baseOptions.length ? baseOptions[0].outerHTML : '<option value="">Todos los clientes</option>';
                    sel.innerHTML = first;

                    rows.forEach(c => {
                        const opt = document.createElement('option');
                        opt.value = String(c.id);
                        opt.textContent = c.name || ('Cliente ' + c.id);
                        sel.appendChild(opt);
                    });

                    // restaurar si existe
                    if (cur) sel.value = cur;
                }

                fillSel(selClientePend);
                fillSel(selClienteHist);
                fillSel(kpiCliente);

            } catch (e) {
                console.error('Error loadClientesForFilters', e);
                // no bloquea
            }
        }

        function destroyChartSafe(ch) {
            try { if (ch && typeof ch.destroy === 'function') ch.destroy(); } catch (e) { }
        }

        function renderTopProductsPie(items) {
            if (!chartCanvas || typeof Chart === 'undefined') return;

            destroyChartSafe(chartTop);
            chartTop = null;

            const labels = (items || []).map(x => x.name || 'Producto');
            const data = (items || []).map(x => Number(x.qty || 0));

            if (!labels.length) {
                if (chartNote) chartNote.textContent = 'Sin datos para el periodo/filtro seleccionado.';
                const ctx = chartCanvas.getContext('2d');
                if (ctx) ctx.clearRect(0, 0, chartCanvas.width, chartCanvas.height);
                return;
            }

            if (chartNote) chartNote.textContent = 'Basado en suma de cantidades solicitadas del mes.';

            chartTop = new Chart(chartCanvas, {
                type: 'pie',
                data: {
                    labels,
                    datasets: [{
                        data
                    }]
                },
                options: {
                    responsive: true,
                    plugins: {
                        legend: { position: 'bottom' },
                        tooltip: {
                            callbacks: {
                                label: (ctx) => {
                                    const v = ctx.parsed;
                                    const total = ctx.dataset.data.reduce((a, b) => a + b, 0);
                                    const pct = total ? ((v / total) * 100) : 0;
                                    return ` ${ctx.label}: ${v} (${pct.toFixed(1)}%)`;
                                }
                            }
                        }
                    }
                }
            });
        }

        const loadVentasPendientes = async () => {
            if (!cont) return;
            cont.innerHTML = 'Cargando...';

            const customer_id = selClientePend?.value ? Number(selClientePend.value) : 0;

            try {
                const r = await api('pedidos_admin', { status: 'pendiente', customer_id: customer_id || undefined }, 'GET');
                if (!r.ok) throw new Error(r.error || 'Error');

                const rows = r.rows || [];
                if (!rows.length) {
                    cont.innerHTML = '<p class="text-muted">Sin solicitudes pendientes (según filtro).</p>';
                    return;
                }

                let html = '<div class="list-group">';
                rows.forEach(p => {
                    const addrEnc = encodeURIComponent(p.delivery_address || '');
                    const itemsEnc = encodeURIComponent(p.items_desc || '');
                    const horaTxt = p.delivery_time ? ` · ${escapeHTML(p.delivery_time)}` : '';

                    html += `
            <div class="list-group-item">
              <div class="d-flex justify-content-between align-items-start">
                <div>
                  <div><strong>Pedido #${escapeHTML(p.id)}</strong> — ${escapeHTML(p.status)}</div>
                  <div class="small">Cliente: ${escapeHTML(p.cliente)} (${escapeHTML(p.phone)})</div>
                  <div class="small">Entrega: ${escapeHTML(p.delivery_date)}${horaTxt}</div>
                  <div class="small">Dirección: ${escapeHTML(p.delivery_address || '')}</div>
                  <div class="small">Piezas solicitadas: ${escapeHTML(p.qty)}</div>
                </div>
                <div class="text-end">
                  <button class="btn btn-sm btn-primary btn-ver-pedido"
                    data-id="${escapeHTML(p.id)}"
                    data-status="${escapeHTML(p.status)}"
                    data-sales_order="${escapeHTML('PW-' + String(p.id || '').trim())}"
                    data-cliente="${escapeHTML(p.cliente || '')}"
                    data-phone="${escapeHTML(p.phone || '')}"
                    data-address="${addrEnc}"
                    data-qty="${escapeHTML(p.qty)}"
                    data-fecha="${escapeHTML(p.delivery_date || '')}"
                    data-hora="${escapeHTML(p.delivery_time || '')}"
                    data-items="${itemsEnc}">
                    Ver pedido
                  </button>
                </div>
              </div>
            </div>`;
                });
                html += '</div>';
                cont.innerHTML = html;

            } catch (e) {
                cont.innerHTML = '<div class="alert alert-danger">' + escapeHTML(e.message) + '</div>';
            }
        };

        cont?.addEventListener('click', (ev) => {
            const btn = ev.target.closest('.btn-ver-pedido');
            if (!btn || !modalVentas) return;

            pedidoActual = {
                id: btn.dataset.id,
                status: btn.dataset.status,
                sales_order: btn.dataset.sales_order || '',
                cliente: btn.dataset.cliente,
                phone: btn.dataset.phone,
                address: decodeURIComponent(btn.dataset.address || ''),
                qty: btn.dataset.qty,
                fecha: btn.dataset.fecha,
                hora: btn.dataset.hora || '',
                items_desc: decodeURIComponent(btn.dataset.items || '')
            };

            if (infoPedido) {
                const horaLinea = pedidoActual.hora ? `<div>Hora aproximada: ${escapeHTML(pedidoActual.hora)}</div>` : '';
                const itemsHtml = pedidoActual.items_desc ? escapeHTML(pedidoActual.items_desc) : '—';

                infoPedido.innerHTML = `
          <div><strong>Pedido #${escapeHTML(pedidoActual.id)}</strong></div>
          <div>Cliente: ${escapeHTML(pedidoActual.cliente)} (${escapeHTML(pedidoActual.phone)})</div>
          <div>Entrega: ${escapeHTML(pedidoActual.fecha)}</div>
          ${horaLinea}
          <div>Dirección: ${escapeHTML(pedidoActual.address)}</div>
          <div>Piezas totales: ${escapeHTML(pedidoActual.qty)}</div>
          <div class="mt-2"><strong>Detalle de artículos:</strong></div>
          <div>${itemsHtml}</div>
        `;
            }

            if (inputOC) inputOC.value = pedidoActual.sales_order || '';
            if (selectStatus) selectStatus.value = pedidoActual.status || 'pendiente';
            if (inputHora) inputHora.value = pedidoActual.hora || '';
            if (inputFecha) inputFecha.value = pedidoActual.fecha || '';

            // Carga detalle (articulos) para edicion
            if (typeof loadPedidoDetalleVentas === 'function') loadPedidoDetalleVentas(pedidoActual.id);

            modalVentas.show();
        });

        btnDescPDF?.addEventListener('click', () => {
            if (!pedidoActual) return toast('Primero selecciona un pedido');
            const url = `orden_entrega_pdf.php?id=${encodeURIComponent(pedidoActual.id)}`;
            window.open(url, '_blank', 'noopener');
        });

        btnGuardarOC?.addEventListener('click', async () => {
            if (!pedidoActual) return toast('No hay pedido seleccionado');

            const sales_order = (inputOC?.value || '').trim();
            const status = selectStatus?.value || 'autorizada';
            const delivery_time = (inputHora?.value || '').trim();
            const delivery_date = (inputFecha?.value || '').trim();

            const items = [];
            (itemsEditor?.querySelectorAll('[data-product-id]') || []).forEach((row) => {
                const pid = Number(row.dataset.productId || 0);
                const input = row.querySelector('input');
                const qty = Number(input ? input.value : 0);
                if (pid > 0) items.push({ product_id: pid, qty: (isFinite(qty) ? qty : 0) });
            });

            try {
                const r = await api('registrar_ov', {
                    order_id: Number(pedidoActual.id),
                    sales_order,
                    status,
                    delivery_time,
                    delivery_date: (delivery_date || undefined),
                    items: (items && items.length ? items : undefined)
                }, 'POST');

                if (!r.ok) throw new Error(r.error || 'Error al registrar OC');

                toast('Cambios guardados');
                modalVentas.hide();
                loadVentasPendientes();
            } catch (e) {
                toast(e.message);
            }
        });

        btnVerHist?.addEventListener('click', async () => {
            if (!histDesde?.value || !histHasta?.value) {
                toast('Selecciona el rango de fechas');
                return;
            }
            if (!histRes) return;

            const customer_id = selClienteHist?.value ? Number(selClienteHist.value) : 0;

            histRes.innerHTML = 'Cargando historial...';

            try {
                const r = await api('historial_pedidos', {
                    desde: histDesde.value,
                    hasta: histHasta.value,
                    customer_id: customer_id || undefined
                }, 'GET');

                if (!r.ok) throw new Error(r.error || 'Error en historial');

                const rows = r.rows || [];
                if (!rows.length) {
                    histRes.innerHTML = '<p class="text-muted mb-0">Sin registros en el rango seleccionado (según filtro).</p>';
                    return;
                }

                let html = `
          <div class="table-responsive">
          <table class="table table-sm table-striped align-middle mb-0">
            <thead>
              <tr>
                <th>Fecha alta</th>
                <th>Pedido</th>
                <th>OC</th>
                <th>Cliente</th>
                <th>Teléfono</th>
                <th>Dirección</th>
                <th>Piezas</th>
                <th>Estatus</th>
                <th>Fecha entrega</th>
                <th></th>
              </tr>
            </thead>
            <tbody>
        `;
                rows.forEach(rw => {
                    const safeAddr = escapeHTML((rw.delivery_address || '').toString());
                    html += `
            <tr>
              <td>${escapeHTML(rw.created_at)}</td>
              <td>#${escapeHTML(rw.id)}</td>
              <td>${escapeHTML(rw.sales_order || '')}</td>
              <td>${escapeHTML(rw.cliente)}</td>
              <td>${escapeHTML(rw.phone)}</td>
              <td>${safeAddr}</td>
              <td>${escapeHTML(rw.qty)}</td>
              <td>${escapeHTML(rw.status)}</td>
              <td>${escapeHTML(rw.delivery_date)}</td>
              <td>
                <button class="btn btn-sm btn-outline-primary"
                        onclick="window.open('orden_entrega_pdf.php?id=${encodeURIComponent(rw.id)}','_blank','noopener')">
                  Ver PDF
                </button>
              </td>
            </tr>`;
                });
                html += '</tbody></table></div>';
                histRes.innerHTML = html;

            } catch (e) {
                histRes.innerHTML = '<div class="alert alert-danger mb-0">' + escapeHTML(e.message) + '</div>';
            }
        });

        btnExpHist?.addEventListener('click', () => {
            if (!histDesde?.value || !histHasta?.value) {
                toast('Selecciona el rango de fechas');
                return;
            }
            const customer_id = selClienteHist?.value ? String(selClienteHist.value) : '';
            let url = `export_historial.php?desde=${encodeURIComponent(histDesde.value)}&hasta=${encodeURIComponent(histHasta.value)}`;
            if (customer_id) url += `&customer_id=${encodeURIComponent(customer_id)}`;
            window.location.href = url;
        });

        async function loadVentasKpis() {
            const ym = (kpiMes?.value || '').trim() || new Date().toISOString().slice(0, 7);
            const customer_id = kpiCliente?.value ? Number(kpiCliente.value) : 0;

            // placeholders
            if (kpiPend) kpiPend.textContent = '—';
            if (kpiAut) kpiAut.textContent = '—';
            if (kpiEnt) kpiEnt.textContent = '—';
            if (kpiRech) kpiRech.textContent = '—';
            if (kpiRechDet) kpiRechDet.textContent = '';

            try {
                const r = await api('ventas_dashboard', { ym, customer_id: customer_id || undefined }, 'GET');
                if (!r.ok) throw new Error(r.error || 'Error KPIs');

                const k = r.kpis || {};
                if (kpiPend) kpiPend.textContent = String(k.pendientes ?? 0);
                if (kpiAut) kpiAut.textContent = String(k.autorizados ?? 0);
                if (kpiEnt) kpiEnt.textContent = String(k.entregados ?? 0);
                if (kpiRech) kpiRech.textContent = String(k.rechazados ?? 0);

                if (kpiRechDet) {
                    if (k.rechazo_by_supported) {
                        kpiRechDet.textContent = `Cliente: ${k.rechazados_by_cliente ?? 0} · Ventas: ${k.rechazados_by_ventas ?? 0}`;
                    } else {
                        kpiRechDet.textContent = 'Cliente/Ventas: N/D (sin columna de origen de rechazo)';
                    }
                }

                renderTopProductsPie(r.top_products || []);
            } catch (e) {
                console.error('Error KPIs', e);
                if (kpiRechDet) kpiRechDet.textContent = `Error KPIs: ${e.message}`;
                renderTopProductsPie([]);
            }
        }

        // Eventos filtros
        selClientePend?.addEventListener('change', () => loadVentasPendientes());
        btnRefPend?.addEventListener('click', () => loadVentasPendientes());

        kpiMes?.addEventListener('change', () => loadVentasKpis());
        kpiCliente?.addEventListener('change', () => loadVentasKpis());
        btnRefKpi?.addEventListener('click', () => loadVentasKpis());

        // Inicial
        (async () => {
            await loadClientesForFilters();
            await loadVentasPendientes();
            await loadVentasKpis();
        })();
    }
});
