<?php
// includes/phone_helper.php
// Normalización de teléfonos a E.164 según país.
// Reglas del proyecto:
// - MX: +521 + 10 dígitos
// - US: +1   + 10 dígitos

declare(strict_types=1);

function ph_starts_with(string $haystack, string $needle): bool {
    if ($needle === '') return true;
    return substr($haystack, 0, strlen($needle)) === $needle;
}

function ph_digits_only(string $s): string {
    return preg_replace('/\D+/', '', $s) ?? '';
}

/**
 * Normaliza a E.164.
 * MX:
 *  - 10 dígitos => +521 + 10
 *  - 12 dígitos iniciando 52 => +521 + últimos 10 (fuerza móvil con 1)
 *  - 13 dígitos iniciando 521 => +521 + 10
 *  - si ya viene con +, se respeta como base y se “fuerza” a +521 si es MX y viene +52 sin 1.
 * US:
 *  - 10 dígitos => +1 + 10
 *  - 11 dígitos iniciando 1 => +1 + 10
 */
function normalize_phone_e164(string $country, string $raw): string {
    $country = strtoupper(trim($country ?: 'MX'));
    $raw = trim((string)$raw);
    if ($raw === '') return '';

    // Si ya viene con +, limpiamos a dígitos y aplicamos reglas país.
    if ($raw[0] === '+') {
        $d = ph_digits_only($raw);
        if (strlen($d) < 8) return '';

        if ($country === 'MX') {
            // Si viene +52XXXXXXXXXX (12 dígitos), lo convertimos a +521XXXXXXXXXX
            if (ph_starts_with($d, '52') && strlen($d) === 12) {
                return '+521' . substr($d, 2);
            }
            // Si viene +521XXXXXXXXXX (13 dígitos), ok
            if (ph_starts_with($d, '521') && strlen($d) === 13) {
                return '+' . $d;
            }
            // Si viene otra cosa rara en MX, devolvemos lo que venga con +
            return '+' . $d;
        }

        if ($country === 'US') {
            // +1XXXXXXXXXX (11 dígitos)
            if (ph_starts_with($d, '1') && strlen($d) === 11) return '+' . $d;
            // +XXXXXXXXXX (10) -> asumimos US => +1
            if (strlen($d) === 10) return '+1' . $d;
            return '+' . $d;
        }

        return '+' . $d;
    }

    // Sin +: sólo dígitos
    $d = ph_digits_only($raw);
    if ($d === '') return '';

    if ($country === 'MX') {
        if (strlen($d) === 10) return '+521' . $d;
        if (strlen($d) === 12 && ph_starts_with($d, '52')) return '+521' . substr($d, 2);
        if (strlen($d) === 13 && ph_starts_with($d, '521')) return '+' . $d;
        return '';
    }

    if ($country === 'US') {
        if (strlen($d) === 10) return '+1' . $d;
        if (strlen($d) === 11 && ph_starts_with($d, '1')) return '+' . $d;
        return '';
    }

    return '';
}

/**
 * Retorna los últimos 10 dígitos “nacionales” (útil para guardar/validar),
 * o vacío si no se puede obtener 10 dígitos.
 */
function phone_national_10(string $country, string $raw): string {
    $d = ph_digits_only($raw);
    if (strlen($d) < 10) return '';
    return substr($d, -10);
}
