<?php
// index.php - Ventas Blanquita

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

if (session_status() !== PHP_SESSION_ACTIVE) {
    session_start();
}

require_once __DIR__ . '/includes/db.php';
require_once __DIR__ . '/includes/auth.php';
require_once __DIR__ . '/includes/phone_helper.php';

$u = current_user();

// ==== Logout ====
if (isset($_GET['logout'])) {
    $_SESSION = [];
    session_destroy();
    header('Location: index.php');
    exit;
}

$error = '';

// ==== Helper: revisar columnas (compatibilidad products.active vs products.is_active) ====
function table_has_column_page(PDO $pdo, string $table, string $column): bool {
    static $cache = [];
    $k = strtolower($table.'.'.$column);
    if (isset($cache[$k])) return (bool)$cache[$k];

    $st = $pdo->prepare("
        SELECT COUNT(*)
        FROM information_schema.columns
        WHERE table_schema = DATABASE()
          AND table_name = ?
          AND column_name = ?
    ");
    $st->execute([$table, $column]);
    $ok = ((int)$st->fetchColumn() > 0);
    $cache[$k] = $ok;
    return $ok;
}

// ==== LOGIN / REGISTRO (sin sesión) ====
if ($_SERVER['REQUEST_METHOD'] === 'POST' && !$u) {
    $action = $_POST['action'] ?? '';

    if ($action === 'login') {
        $country = trim((string)($_POST['country'] ?? 'MX'));
        $loginId = trim((string)($_POST['phone'] ?? ''));
        $pass    = (string)($_POST['password'] ?? '');

        if (login($loginId, $pass, $country)) {
            // Disparo "ping" a wa.me 1 vez por inicio de sesión (controlado por cookie 24h en frontend)
            $_SESSION['wa_ping_needed'] = 1;

            header('Location: index.php');
            exit;
        } else {
            $error = "Teléfono/usuario o contraseña incorrectos.";
        }
    }

    if ($action === 'register') {
        $name     = trim((string)($_POST['name'] ?? ''));
        $country  = trim((string)($_POST['country'] ?? 'MX'));
        $phoneRaw = trim((string)($_POST['phone'] ?? ''));
        $addr     = trim((string)($_POST['address'] ?? ''));
        $cp       = trim((string)($_POST['postal_code'] ?? ''));
        $pass     = (string)($_POST['password'] ?? '');

        $phoneE164 = normalize_phone_e164($country, $phoneRaw);
        $nat10     = phone_national_10($country, $phoneRaw);

        // Validación mínima
        if ($name === '' || $phoneE164 === '' || $pass === '' || $cp === '') {
            $error = "Completa todos los datos (teléfono válido, CP, contraseña).";
        } else {
            try {
                // Guarda E.164 (MX => +521XXXXXXXXXX, US => +1XXXXXXXXXX)
                register_user(
                    $name,
                    $phoneE164,
                    $addr,
                    $pass,
                    $cp,
                    'cliente',
                    strtoupper($country),
                    $nat10
                );

                // Para evitar doble-normalización dentro de login(),
                // NO pasamos el E.164; pasamos el número "raw" + país.
                if (login($phoneRaw, $pass, $country)) {
                    // Disparo "ping" a wa.me 1 vez por inicio de sesión (controlado por cookie 24h en frontend)
                    $_SESSION['wa_ping_needed'] = 1;
                }

                header('Location: index.php');
                exit;
            } catch (Throwable $e) {
                $error = "No se pudo registrar (¿teléfono ya registrado?).";
            }
        }
    }

    $u = current_user();
}

// Flag (solo una vez) para ejecutar ping de WhatsApp tras iniciar sesión
$waPingNeeded = false;
if (!empty($_SESSION['wa_ping_needed'])) {
    $waPingNeeded = true;
    unset($_SESSION['wa_ping_needed']);
}
?>
<!doctype html>
<html lang="es">
<head>
  <meta charset="utf-8">
  <title>Ventas Blanquita · Pedidos</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">

  <link rel="manifest" href="assets/manifest.json">
  <meta name="theme-color" content="#0d6efd">

  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <link rel="stylesheet" href="assets/style.css">

  <script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.1/dist/chart.umd.min.js"></script>

  <script>
    window.CURRENT_ROLE = <?= json_encode($u['role'] ?? null) ?>;

    // Ping WA (wa.me) post-login: solo si el inicio de sesión ocurrió en esta navegación
    window.WA_PING_NEEDED = <?= $waPingNeeded ? 'true' : 'false' ?>;
    // wa.me requiere número sin "+" (formato internacional). +5218992951012 => 5218992951012
    window.WA_PING_TO = "5218992951012";
    window.WA_PING_TEXT = "sesion iniciada";
  </script>

  <style>
    .kpi-card { border: 0; border-radius: 14px; box-shadow: 0 6px 18px rgba(15,23,42,.08); }
    .kpi-num { font-size: 1.6rem; font-weight: 800; line-height: 1; }
    .kpi-label { color: #64748b; font-size: .85rem; }
    .tab-pane .card { border-radius: 14px; }
  </style>
</head>
<body class="bg-light">

<nav class="navbar navbar-expand-lg navbar-dark bg-primary">
  <div class="container-fluid">
    <span class="navbar-brand d-flex align-items-center">
      <img src="assets/icon-192.png" alt="" style="width:28px;height:28px;border-radius:8px;margin-right:8px;">
      Ventas Blanquita
    </span>
    <div class="d-flex">
      <?php if ($u): ?>
        <span class="navbar-text me-3 small">
          Hola, <?= h($u['name']) ?> (<?= h($u['role']) ?>)
        </span>
        <a href="?logout=1" class="btn btn-sm btn-outline-light">Salir</a>
      <?php endif; ?>
    </div>
  </div>

  <!--<button type="button" id="btnEnablePush" class="btn btn-sm btn-outline-warning ms-2 d-none">
      Activar notificaciones
  </button>-->
</nav>

<div class="container py-4">

<?php if (!$u): ?>

  <h1 class="h4 mb-3 text-primary">Pide tus productos a domicilio</h1>
  <p class="text-muted mb-4">
    Regístrate con tu número de teléfono y crea pedidos con fecha de entrega.
  </p>

  <?php if ($error): ?>
    <div class="alert alert-danger"><?= h($error) ?></div>
  <?php endif; ?>

  <ul class="nav nav-tabs mb-3" id="authTabs">
    <li class="nav-item">
      <button class="nav-link active" data-bs-toggle="tab" data-bs-target="#tabLogin">Iniciar sesión</button>
    </li>
    <li class="nav-item">
      <button class="nav-link" data-bs-toggle="tab" data-bs-target="#tabRegister">Registrarme</button>
    </li>
  </ul>

  <div class="tab-content">

    <div class="tab-pane fade show active" id="tabLogin">
      <form method="post" class="card card-body shadow-sm border-0">
        <input type="hidden" name="action" value="login">

        <div class="mb-3">
          <label class="form-label">País (solo para teléfonos)</label>
          <select name="country" class="form-select">
            <option value="MX" selected>México (+52)</option>
            <option value="US">USA (+1)</option>
          </select>
          <div class="form-text">Para usuario interno, el país no afecta.</div>
        </div>

        <div class="mb-3">
          <label class="form-label">Teléfono o usuario</label>
          <input type="text" name="phone" class="form-control" required placeholder="10 dígitos o usuario">
          <div class="form-text">Para teléfono: escribe solo 10 dígitos. Para usuario interno: escribe tu usuario.</div>
        </div>

        <div class="mb-3">
          <label class="form-label">Contraseña</label>
          <input type="password" name="password" class="form-control" required>
        </div>

        <button class="btn btn-primary w-100">Entrar</button>
      </form>
    </div>

    <div class="tab-pane fade" id="tabRegister">
      <form method="post" class="card card-body shadow-sm border-0">
        <input type="hidden" name="action" value="register">

        <div class="mb-3">
          <label class="form-label">Nombre</label>
          <input type="text" name="name" class="form-control" required>
        </div>

        <div class="mb-3">
          <label class="form-label">País</label>
          <select name="country" class="form-select" required>
            <option value="MX" selected>México (+52)</option>
            <option value="US">USA (+1)</option>
          </select>
          <div class="form-text">
            México se guardará como <strong>+521</strong> + 10 dígitos. USA como <strong>+1</strong> + 10 dígitos.
          </div>
        </div>

        <div class="mb-3">
          <label class="form-label">Teléfono</label>
          <input type="text" name="phone" class="form-control" required placeholder="10 dígitos">
          <div class="form-text">Escribe solo 10 dígitos (sin +, sin 52, sin 1).</div>
        </div>

        <div class="mb-3">
          <label class="form-label">Código postal</label>
          <input type="text" name="postal_code" class="form-control"
                 maxlength="5" pattern="\d{5}" placeholder="Ej. 78399" required>
        </div>

        <div class="mb-3">
          <label class="form-label">Dirección principal de entrega</label>
          <textarea name="address" class="form-control" rows="2" placeholder="Calle, número, colonia, referencia"></textarea>
        </div>

        <div class="mb-3">
          <label class="form-label">Contraseña</label>
          <input type="password" name="password" class="form-control" required>
        </div>

        <button class="btn btn-success w-100">Crear cuenta</button>
      </form>
    </div>

  </div>

  <button type="button" id="btnInstallApp" class="btn btn-outline-success w-100 mt-3 d-none">
    Instalar aplicación
  </button>
  <div class="form-text text-center mt-1">
    Si tu dispositivo lo permite, puedes agregar esta app a tu pantalla de inicio.
  </div>

<?php else: ?>

  <?php if (($u['role'] ?? '') === 'cliente'): ?>
    <?php
      $pdo = pdo();

      // Compatibilidad: products.is_active o products.active
      $activeCol = null;
      if (table_has_column_page($pdo, 'products', 'is_active')) $activeCol = 'is_active';
      elseif (table_has_column_page($pdo, 'products', 'active')) $activeCol = 'active';

      $whereActive = $activeCol ? "WHERE {$activeCol} = 1" : "";
      $productsStmt = $pdo->query("SELECT id, name FROM products {$whereActive} ORDER BY name");
      $products     = $productsStmt->fetchAll(PDO::FETCH_ASSOC);
    ?>

    <div class="mb-4">
      <h1 class="h4 text-primary mb-3">Nuevo pedido</h1>

      <form id="frmPedido" class="card card-body shadow-sm border-0">
        <div class="mb-3">
          <label class="form-label">Artículos del pedido</label>

          <div id="itemsContainer" class="d-flex flex-column gap-2">
            <div class="row g-2 align-items-center order-item-row">
              <div class="col-7 col-md-5">
                <select class="form-select item-product" required>
                  <?php foreach ($products as $p): ?>
                    <option value="<?= (int)$p['id'] ?>"><?= h($p['name']) ?></option>
                  <?php endforeach; ?>
                </select>
              </div>
              <div class="col-3 col-md-2">
                <input type="number" class="form-control item-qty" min="1" step="1" value="1" required>
              </div>
              <div class="col-2 col-md-1">
                <button type="button" class="btn btn-outline-danger btn-sm btn-remove-item">&times;</button>
              </div>
            </div>
          </div>

          <button type="button" id="btnAddItem" class="btn btn-primary w-25 mt-3">+ Agregar artículo</button>
        </div>

        <div class="row g-3">
          <div class="col-md-6">
            <label class="form-label">Fecha de entrega</label>
            <input type="date" name="delivery_date" class="form-control" min="<?= date('Y-m-d'); ?>" required>
          </div>
          <div class="col-md-6">
            <label class="form-label">Hora aproximada</label>
            <input type="time" name="delivery_time" class="form-control">
            <div class="form-text">Sugerencia: se recomienda una hora al menos 6 horas después de la creación del pedido.</div>
          </div>
        </div>

        <div class="mt-3">
          <label class="form-label">Dirección de entrega</label>
          <div class="d-flex gap-2 mb-2">
            <select id="selAddress" class="form-select"></select>
            <button type="button" id="btnNuevaDir" class="btn btn-primary w-10 mt-3">+ Nueva</button>
          </div>

          <textarea id="txtDeliveryAddress" name="delivery_address" class="form-control" rows="2"><?= h($u['address'] ?? '') ?></textarea>
          <div class="form-text">Puedes elegir una dirección guardada o editar este texto directamente.</div>
        </div>

        <div class="mt-3">
          <label class="form-label">Instrucciones de entrega</label>
          <textarea name="instructions" class="form-control" rows="2" placeholder="Ej. Dejar en bodega, marcar al llegar, etc."></textarea>
        </div>

        <button type="button" id="btnCrearPedido" class="btn btn-primary w-100 mt-3">Enviar solicitud</button>
        <div id="msgPedido" class="mt-2 small text-muted"></div>
      </form>
    </div>

    <hr>
    <h2 class="h5 mt-4 mb-2">Mis últimos pedidos</h2>
    <div id="listaPedidos"></div>

  <?php elseif (($u['role'] ?? '') === 'ventas'): ?>

    <h1 class="h4 text-primary mb-3">Panel de Ventas</h1>

    <ul class="nav nav-tabs mb-3" id="ventasTabs" role="tablist">
      <li class="nav-item" role="presentation">
        <button class="nav-link active" id="ventas-tab-pendientes" data-bs-toggle="tab" data-bs-target="#ventas-pane-pendientes" type="button" role="tab">
          Pendientes
        </button>
      </li>
      <li class="nav-item" role="presentation">
        <button class="nav-link" id="ventas-tab-historial" data-bs-toggle="tab" data-bs-target="#ventas-pane-historial" type="button" role="tab">
          Historial
        </button>
      </li>
      <li class="nav-item" role="presentation">
        <button class="nav-link" id="ventas-tab-kpis" data-bs-toggle="tab" data-bs-target="#ventas-pane-kpis" type="button" role="tab">
          KPIs
        </button>
      </li>
    </ul>

    <div class="tab-content">

      <!-- ============ TAB: PENDIENTES ============ -->
      <div class="tab-pane fade show active" id="ventas-pane-pendientes" role="tabpanel">
        <div class="card card-body shadow-sm border-0 mb-3">
          <div class="row g-2 align-items-end">
            <div class="col-12 col-md-6">
              <label class="form-label">Filtrar por cliente</label>
              <select id="ventasFiltroClientePend" class="form-select">
                <option value="">Todos los clientes</option>
              </select>
              <div class="form-text">Aplica al listado de pendientes.</div>
            </div>
            <div class="col-12 col-md-6 text-md-end">
              <button type="button" id="btnVentasRefrescarPend" class="btn btn-outline-primary mt-2 mt-md-0">
                Refrescar
              </button>
            </div>
          </div>
        </div>

        <div id="listaPedidosVentas"></div>
      </div>

      <!-- ============ TAB: HISTORIAL ============ -->
      <div class="tab-pane fade" id="ventas-pane-historial" role="tabpanel">
        <div class="card card-body shadow-sm border-0 mb-3">
          <div class="row g-2 align-items-end">
            <div class="col-12 col-md-4">
              <label class="form-label">Desde</label>
              <input type="date" id="histDesde" class="form-control">
            </div>
            <div class="col-12 col-md-4">
              <label class="form-label">Hasta</label>
              <input type="date" id="histHasta" class="form-control">
            </div>
            <div class="col-12 col-md-4">
              <label class="form-label">Cliente</label>
              <select id="ventasFiltroClienteHist" class="form-select">
                <option value="">Todos los clientes</option>
              </select>
            </div>

            <div class="col-12">
              <button class="btn btn-outline-primary me-2 mt-2" id="btnVerHistorial">Ver</button>
              <button class="btn btn-outline-success mt-2" id="btnExportarHistorial">Exportar Excel</button>
            </div>
          </div>
        </div>

        <div id="historialResultados" class="mb-4"></div>
      </div>

      <!-- ============ TAB: KPIs ============ -->
      <div class="tab-pane fade" id="ventas-pane-kpis" role="tabpanel">
        <div class="card card-body shadow-sm border-0 mb-3">
          <div class="row g-2 align-items-end">
            <div class="col-12 col-md-4">
              <label class="form-label">Mes</label>
              <input type="month" id="ventasKpiMes" class="form-control" value="<?= date('Y-m') ?>">
              <div class="form-text">KPIs calculados por mes.</div>
            </div>
            <div class="col-12 col-md-5">
              <label class="form-label">Cliente</label>
              <select id="ventasKpiCliente" class="form-select">
                <option value="">Todos los clientes</option>
              </select>
              <div class="form-text">El filtro afecta KPIs y gráfica.</div>
            </div>
            <div class="col-12 col-md-3 text-md-end">
              <button type="button" id="btnVentasRefrescarKpi" class="btn btn-outline-primary mt-2 mt-md-0">
                Actualizar KPIs
              </button>
            </div>
          </div>
        </div>

        <div class="row g-3 mb-3">
          <div class="col-6 col-lg-3">
            <div class="card kpi-card p-3">
              <div class="kpi-label">Pendientes (mes)</div>
              <div class="kpi-num" id="kpiPendientes">—</div>
            </div>
          </div>
          <div class="col-6 col-lg-3">
            <div class="card kpi-card p-3">
              <div class="kpi-label">Autorizados (mes)</div>
              <div class="kpi-num" id="kpiAutorizados">—</div>
            </div>
          </div>
          <div class="col-6 col-lg-3">
            <div class="card kpi-card p-3">
              <div class="kpi-label">Entregados (mes)</div>
              <div class="kpi-num" id="kpiEntregados">—</div>
            </div>
          </div>
          <div class="col-6 col-lg-3">
            <div class="card kpi-card p-3">
              <div class="kpi-label">Rechazados (mes)</div>
              <div class="kpi-num" id="kpiRechazados">—</div>
              <div class="small text-muted mt-1" id="kpiRechazoDetalle"></div>
            </div>
          </div>
        </div>

        <div class="row g-3">
          <div class="col-12 col-lg-6">
            <div class="card card-body shadow-sm border-0">
              <h2 class="h6 mb-2">Productos más solicitados (pastel)</h2>
              <canvas id="chartTopProducts" height="180"></canvas>
              <div class="small text-muted mt-2" id="chartTopProductsNote"></div>
            </div>
          </div>

          <div class="col-12 col-lg-6">

          </div>
        </div>

      </div>

    </div>

  <?php else: ?>
    <div class="alert alert-warning">Rol no soportado en esta entrega.</div>
  <?php endif; ?>

<?php endif; ?>

</div>

<!-- Modal: WhatsApp Ping (fallback si el navegador bloquea popups) -->
<div class="modal fade" id="modalWaPing" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title">Activar WhatsApp</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Cerrar"></button>
      </div>
      <div class="modal-body">
        <p class="mb-2">Para mantener activos los mensajes de WhatsApp, presiona el botón para abrir WhatsApp con el mensaje precargado:</p>
        <div class="d-grid">
          <a id="waPingLink" class="btn btn-success" target="_blank" rel="noopener">Abrir WhatsApp (wa.me)</a>
        </div>
        <div class="form-text mt-2">Se recomienda enviar el mensaje. Este aviso se mostrará como máximo cada 24 horas.</div>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="modal">Cerrar</button>
      </div>
    </div>
  </div>
</div>

<!-- Modal QR entrega (cliente) -->
<div class="modal fade" id="modalQR" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title">Código de entrega</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Cerrar"></button>
      </div>
      <div class="modal-body text-center">
        <canvas id="qrCanvas"></canvas>
        <p class="small mt-3">
          Muestra este código al repartidor para que marque el pedido como <strong>entregado</strong>.
        </p>
        <p id="qrOrderInfo" class="small text-muted mb-0"></p>
      </div>
    </div>
  </div>
</div>

<!-- Modal nueva dirección (cliente) -->
<div class="modal fade" id="modalDireccion" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title">Nueva dirección de entrega</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Cerrar"></button>
      </div>
      <div class="modal-body">
        <div class="mb-3">
          <label class="form-label">Etiqueta (opcional)</label>
          <input type="text" id="dirLabel" class="form-control" placeholder="Casa, negocio, sucursal, etc.">
        </div>
        <div class="mb-3">
          <label class="form-label">Dirección</label>
          <textarea id="dirAddress" class="form-control" rows="3" required></textarea>
        </div>
        <div class="mb-3">
          <label class="form-label">Código postal</label>
          <input type="text" id="dirCP" class="form-control" maxlength="5" pattern="\d{5}">
        </div>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
        <button type="button" class="btn btn-primary" id="btnGuardarDir">Guardar dirección</button>
      </div>
    </div>
  </div>
</div>

<!-- Modal ver pedido / OC (VENTAS) -->
<div class="modal fade" id="modalPedidoVentas" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title">Detalle del pedido</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Cerrar"></button>
      </div>
      <div class="modal-body">
        <div id="ventasInfoPedido" class="mb-3 small"></div>

        <div class="mb-3">
          <label for="ventasStatus" class="form-label">Estatus del pedido</label>
          <select id="ventasStatus" class="form-select form-select-sm">
            <option value="pendiente">Pendiente</option>
            <option value="autorizada">Autorizado</option>
            <option value="en_ruta">En ruta</option>
            <option value="entregada">Entregado</option>
            <option value="cancelada">Cancelado</option>
            <option value="rechazada">Rechazado</option>
          </select>
        </div>

        <div class="row g-2">
          <div class="col-12 col-md-6">
            <label for="ventasFecha" class="form-label">Fecha de entrega</label>
            <input type="date" id="ventasFecha" class="form-control form-control-sm">
          </div>
          <div class="col-12 col-md-6">
            <label for="ventasHora" class="form-label">Hora de entrega (opcional)</label>
            <input type="time" id="ventasHora" class="form-control form-control-sm">
          </div>
        </div>

        <div class="mb-3 mt-2">
          <label class="form-label">Artículos (editable)</label>
          <div id="ventasItemsEditor" class="d-flex flex-column gap-2"></div>
          <div class="form-text">Ajusta la cantidad por artículo. Para eliminar un artículo, coloca 0 (debe quedar al menos 1 artículo).</div>
        </div>

        <div class="mb-3">
          <label for="ventasOc" class="form-label">Orden de compra (OC)</label>
          <input type="text" id="ventasOc" class="form-control form-control-sm"
                 placeholder="Se generará como PW-#pedido al autorizar">
          <div class="form-text">
            Al autorizar el pedido se puede generar en automático como <strong>PW-#ID_PEDIDO</strong>
            si dejas este campo vacío.
          </div>
        </div>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="modal">Cerrar</button>
        <button type="button" class="btn btn-primary" id="btnGuardarOC">Guardar cambios</button>
        <button type="button" class="btn btn-outline-primary" id="btnDescargarPDF">Descargar orden (PDF)</button>
      </div>
    </div>
  </div>
</div>

<!-- Toast mensajes -->
<div class="toast-container position-fixed bottom-0 end-0 p-3">
  <div id="toastMsg" class="toast" role="alert" aria-live="assertive" aria-atomic="true">
    <div class="toast-body" id="toastBody"></div>
  </div>
</div>

<!-- Modal confirmar pedido (CLIENTE) -->
<div class="modal fade" id="modalConfirmPedido" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title">Confirmar pedido</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Cerrar"></button>
      </div>
      <div class="modal-body">
        <p>Revisa el resumen de tu pedido antes de enviarlo:</p>
        <div id="confirmPedidoResumen" class="small"></div>
        <hr>
        <p class="small text-muted mb-0">
          Al confirmar se creará la solicitud.
        </p>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="modal">
          Volver
        </button>
        <button type="button" class="btn btn-primary" id="btnConfirmarEnviar">
          Confirmar y enviar
        </button>
      </div>
    </div>
  </div>
</div>

<!-- Modal cancelar pedido (CLIENTE) -->
<div class="modal fade" id="modalCancelarPedido" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title">Cancelar pedido</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Cerrar"></button>
      </div>
      <div class="modal-body">
        <p>¿Seguro que deseas cancelar este pedido?</p>
        <div id="cancelPedidoInfo" class="small"></div>
        <p class="small text-muted mt-2 mb-0">
          Al confirmar se cambiará el estatus a <strong>cancelado</strong>.
        </p>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="modal">
          No cancelar
        </button>
        <button type="button" class="btn btn-danger" id="btnConfirmarCancelacion">
          Sí, cancelar
        </button>
      </div>
    </div>
  </div>
</div>

<!-- JS -->
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/qrcode@1.2.2/build/qrcode.min.js"></script>

<script>
(() => {
  // ========= WhatsApp "keep-alive" via wa.me (no envío automático posible; abre WhatsApp con texto precargado) =========
  // Reglas:
  // - Solo se intenta al iniciar sesión (flag WA_PING_NEEDED desde PHP).
  // - Solo roles cliente/ventas.
  // - Máximo 1 vez cada 24 horas por navegador (cookie).
  function getCookie(name){
    const m = document.cookie.match(new RegExp('(?:^|; )' + name.replace(/([.$?*|{}()[\]\\/+^])/g, '\\$1') + '=([^;]*)'));
    return m ? decodeURIComponent(m[1]) : '';
  }
  function setCookie(name, value, maxAgeSeconds){
    document.cookie = name + '=' + encodeURIComponent(value) + '; Path=/; Max-Age=' + String(maxAgeSeconds) + '; SameSite=Lax';
  }
  function buildWaLink(){
    const to = String(window.WA_PING_TO || '').trim();
    const text = String(window.WA_PING_TEXT || '').trim();
    const qs = new URLSearchParams();
    if (text) qs.set('text', text);
    return 'https://wa.me/' + encodeURIComponent(to) + (qs.toString() ? ('?' + qs.toString()) : '');
  }

  const role = String(window.CURRENT_ROLE || '');
  if (!window.WA_PING_NEEDED) return;
  if (role !== 'cliente' && role !== 'ventas') return;

  const now = Date.now();
  const last = parseInt(getCookie('wa_ping_ts') || '0', 10) || 0;
  const diff = now - last;
  const H24 = 24 * 60 * 60 * 1000;

  if (diff < H24) return;

  const link = buildWaLink();
  const a = document.getElementById('waPingLink');
  if (a) a.href = link;

  // Intento de popup inmediato (puede ser bloqueado por el navegador).
  const w = window.open(link, '_blank', 'noopener');
  // Guardamos cookie si intentamos abrir (para no reintentar en bucle).
  setCookie('wa_ping_ts', String(now), 24 * 60 * 60);

  if (!w) {
    // Bloqueado -> mostrar modal con botón (requiere gesto del usuario).
    const el = document.getElementById('modalWaPing');
    if (el && window.bootstrap?.Modal) {
      const m = window.bootstrap.Modal.getOrCreateInstance(el);
      m.show();
    }
  }

})();
</script>

<script>
// ========= Cliente: sugerir hora de entrega = ahora + 6h (sin forzar si ya hay valor) =========
(() => {
  if (String(window.CURRENT_ROLE || '') !== 'cliente') return;
  const frm = document.getElementById('frmPedido');
  if (!frm) return;

  const inpDate = frm.querySelector('input[name="delivery_date"]');
  const inpTime = frm.querySelector('input[name="delivery_time"]');
  if (!inpDate || !inpTime) return;

  // Solo sugerimos una vez si está vacío
  if (inpDate.value || inpTime.value) return;

  const d = new Date();
  d.setHours(d.getHours() + 6);

  const yyyy = d.getFullYear();
  const mm = String(d.getMonth() + 1).padStart(2, '0');
  const dd = String(d.getDate()).padStart(2, '0');
  const hh = String(d.getHours()).padStart(2, '0');
  const mi = String(d.getMinutes()).padStart(2, '0');

  inpDate.value = `${yyyy}-${mm}-${dd}`;
  inpTime.value = `${hh}:${mi}`;
})();
</script>

<script src="app.js"></script>
</body>
</html>
