<?php
// push_helper.php
// Envío de Web Push con Minishlink/WebPush
declare(strict_types=1);

require_once __DIR__ . '/includes/db.php';

// ======= Cargar librería WebPush (vendor) con fallback =======
$autoload = null;

$candidates = [
    __DIR__ . '/vendor/autoload.php',      // vendor dentro del proyecto
    __DIR__ . '/../vendor/autoload.php',   // vendor a nivel public_html
    __DIR__ . '/../../vendor/autoload.php' // un nivel más arriba
];

foreach ($candidates as $c) {
    if (is_file($c)) { $autoload = $c; break; }
}

if (!$autoload) {
    function pushlib_available(): bool { return false; }
    function getVapidPublicKey(): string { return ''; }
    function sendPushToUser(PDO $pdo, int $userId, string $title, string $body, array $data = []): bool { return false; }
    function sendPushToRole(PDO $pdo, string $role, string $title, string $body, array $data = []): bool { return false; }
    return;
}

require_once $autoload;

use Minishlink\WebPush\WebPush;
use Minishlink\WebPush\Subscription;

/** Librería disponible */
function pushlib_available(): bool {
    return class_exists(WebPush::class) && class_exists(Subscription::class);
}

/** Lee JSON desde archivo (si no existe, null). */
function loadKeyFile(string $file): ?array {
    if (!is_file($file)) return null;
    $raw = file_get_contents($file);
    if ($raw === false) return null;
    $j = json_decode($raw, true);
    return is_array($j) ? $j : null;
}

/**
 * Config VAPID:
 * - config/push_vapid.json recomendado:
 *   { "publicKey":"...", "privateKey":"...", "subject":"mailto:..." }
 */
function getVapidConfig(): array {
    $file = __DIR__ . '/config/push_vapid.json';
    $cfg = loadKeyFile($file);

    if (!$cfg) {
        // Fallback (solo publicKey). OJO: sin privateKey real NO se puede ENVIAR push.
        $cfg = [
            'publicKey'  => 'BN5QlWO3yM0renVZfXMY9QId7pOdpEaSkNbz5LtFFCEJefDaYoJur-hDix3DWuexZJtSqtcBWKjbwG7vKw_PNsA',
            'privateKey' => '',
            'subject'    => 'mailto:soporte@tu-dominio.com'
        ];
    }

    $cfg['publicKey']  = (string)($cfg['publicKey'] ?? '');
    $cfg['privateKey'] = (string)($cfg['privateKey'] ?? '');
    $cfg['subject']    = (string)($cfg['subject'] ?? 'mailto:soporte@tu-dominio.com');

    return $cfg;
}

/** Public key para que app.js la obtenga desde api.php */
function getVapidPublicKey(): string {
    $cfg = getVapidConfig();
    return trim($cfg['publicKey']);
}

/** Valida si tenemos lo mínimo para enviar */
function vapid_ready(): bool {
    $cfg = getVapidConfig();
    return (trim($cfg['publicKey']) !== '' && trim($cfg['privateKey']) !== '');
}

/** Instancia WebPush */
function getWebPushInstance(): WebPush {
    $cfg = getVapidConfig();

    $auth = [
        'VAPID' => [
            'subject'    => $cfg['subject'],
            'publicKey'  => $cfg['publicKey'],
            'privateKey' => $cfg['privateKey'],
        ],
    ];

    $webPush = new WebPush($auth);
    $webPush->setReuseVAPIDHeaders(true);
    return $webPush;
}

/**
 * Envía notificación a TODAS las suscripciones del usuario.
 * Nota: firma compatible con api.php => (PDO $pdo, int $userId, ...)
 */
function sendPushToUser(PDO $pdo, int $userId, string $title, string $body, array $data = []): bool {
    if (!pushlib_available()) return false;
    if (!vapid_ready()) return false;

    $stmt = $pdo->prepare("SELECT id, endpoint, p256dh, auth FROM push_subscriptions WHERE user_id = ? ORDER BY id DESC");
    $stmt->execute([$userId]);
    $subs = $stmt->fetchAll(PDO::FETCH_ASSOC);
    if (!$subs) return false;

    $webPush = getWebPushInstance();
    $payload = json_encode([
        'title' => $title,
        'body'  => $body,
        'data'  => $data
    ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

    foreach ($subs as $s) {
        try {
            $subscription = Subscription::create([
                'endpoint' => $s['endpoint'],
                'publicKey' => $s['p256dh'],
                'authToken' => $s['auth'],
            ]);
            $webPush->queueNotification($subscription, $payload);
        } catch (Throwable $e) {
            // Ignora suscripciones corruptas
        }
    }

    foreach ($webPush->flush() as $report) {
        // Limpieza opcional: si endpoint ya no existe (410/404), borrarlo
        try {
            if (method_exists($report, 'isSuccess') && !$report->isSuccess()) {
                $req = method_exists($report, 'getRequest') ? $report->getRequest() : null;
                $code = method_exists($report, 'getResponse') && $report->getResponse()
                    ? $report->getResponse()->getStatusCode()
                    : null;

                if ($code === 404 || $code === 410) {
                    // intenta recuperar endpoint del request
                    if ($req && method_exists($req, 'getUri')) {
                        $endpoint = (string)$req->getUri();
                        $del = $pdo->prepare("DELETE FROM push_subscriptions WHERE endpoint = ?");
                        $del->execute([$endpoint]);
                    }
                }
            }
        } catch (Throwable $e) {}
    }

    return true;
}

/** Envía notificación a todos los usuarios con un rol */
function sendPushToRole(PDO $pdo, string $role, string $title, string $body, array $data = []): bool {
    if (!pushlib_available()) return false;
    if (!vapid_ready()) return false;

    $stmt = $pdo->prepare("SELECT id FROM users WHERE role = ?");
    $stmt->execute([$role]);
    $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
    if (!$users) return false;

    $ok = false;
    foreach ($users as $u) {
        $ok = sendPushToUser($pdo, (int)$u['id'], $title, $body, $data) || $ok;
    }
    return $ok;
}
