// service-worker.js - Ventas Blanquita
// Cache SOLO de estáticos + soporte Push
// IMPORTANTE: incrementa versión para invalidar caché previo
const CACHE_NAME = 'ventasapp-v8';

// SOLO estáticos locales
const ASSETS = [
    './app.js',
    './assets/style.css',
    './assets/icon-192.png',
    './assets/icon-512.png',
    './assets/manifest.json'
];

// INSTALL: cachea estáticos (tolerante a fallos)
self.addEventListener('install', (event) => {
    event.waitUntil((async () => {
        const cache = await caches.open(CACHE_NAME);
        const results = await Promise.allSettled(ASSETS.map(a => cache.add(a)));
        // No rompemos la instalación si un asset falla
        results.forEach(r => { /* noop */ });
        await self.skipWaiting();
    })());
});

// ACTIVATE: limpia cachés viejos
self.addEventListener('activate', (event) => {
    event.waitUntil((async () => {
        const keys = await caches.keys();
        await Promise.all(keys.map(k => (k !== CACHE_NAME ? caches.delete(k) : Promise.resolve())));
        await self.clients.claim();
    })());
});

// FETCH: solo cache para estáticos; NO tocar PHP ni api.php
self.addEventListener('fetch', (event) => {
    const req = event.request;

    if (req.method !== 'GET') return;

    const url = new URL(req.url);

    // No cachear backend
    if (url.pathname.endsWith('.php') || url.pathname.includes('api.php')) return;

    // Solo mismo origen
    if (url.origin !== self.location.origin) return;

    event.respondWith((async () => {
        const cached = await caches.match(req);

        try {
            const res = await fetch(req);
            if (res && res.status === 200) {
                const copy = res.clone();
                const cache = await caches.open(CACHE_NAME);
                await cache.put(req, copy);
            }
            return res;
        } catch (e) {
            if (cached) return cached;
            throw e;
        }
    })());
});

// PUSH
self.addEventListener('push', (event) => {
    let data = {};
    if (event.data) {
        try { data = event.data.json(); }
        catch (e) { data = { body: event.data.text() }; }
    }

    const title = data.title || 'Notificación';
    const options = {
        body: data.body || '',
        icon: './assets/icon-192.png',
        badge: './assets/icon-192.png',
        data: data.data || {}
    };

    event.waitUntil(self.registration.showNotification(title, options));
});

// CLICK notificación
self.addEventListener('notificationclick', (event) => {
    event.notification.close();

    const urlToOpen = (event.notification.data && event.notification.data.url)
        ? event.notification.data.url
        : './';

    event.waitUntil((async () => {
        const clientList = await clients.matchAll({ type: 'window', includeUncontrolled: true });

        for (const client of clientList) {
            if ('focus' in client) {
                await client.navigate(urlToOpen);
                return client.focus();
            }
        }
        return clients.openWindow(urlToOpen);
    })());
});
