<?php
// ventasApp/wa_test.php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');

function fail(string $code, string $detail, int $http = 400): void {
  http_response_code($http);
  echo json_encode(["ok" => false, "error" => $code, "detail" => $detail], JSON_UNESCAPED_UNICODE);
  exit;
}

function base64url_encode(string $bin): string {
  return rtrim(strtr(base64_encode($bin), '+/', '-_'), '=');
}

/**
 * Normaliza un "to" a formato Twilio WhatsApp:
 * - Acepta: "whatsapp:+52...", "+52...", "52...", "4441234567"
 * - Devuelve: "whatsapp:+52XXXXXXXXXX" o '' si no es válido
 */
function normalize_to_whatsapp(string $raw, string $defaultCountry = '+52'): string {
  $raw = trim($raw);
  if ($raw === '') return '';

  // Quitar espacios internos (evita "whatsapp: 52...")
  $raw = preg_replace('/\s+/', '', $raw) ?? '';

  // Si ya trae prefijo whatsapp:
  if (stripos($raw, 'whatsapp:') === 0) {
    $n = substr($raw, 9); // después de "whatsapp:"
    $n = ltrim($n);
    // Si no tiene '+', agrégalo
    if ($n !== '' && $n[0] !== '+') $n = '+' . $n;
    // Solo + y dígitos
    $n = preg_replace('/[^0-9\+]/', '', $n) ?? '';
    return ($n !== '' && $n[0] === '+') ? ('whatsapp:' . $n) : '';
  }

  // Si viene como +E164
  if (strlen($raw) > 0 && $raw[0] === '+') {
    $n = preg_replace('/[^0-9\+]/', '', $raw) ?? '';
    return ($n !== '' && $n[0] === '+') ? ('whatsapp:' . $n) : '';
  }

  // Si viene solo dígitos u otro texto: extraer solo dígitos
  $digits = preg_replace('/\D/', '', $raw) ?? '';
  if ($digits === '') return '';

  // MX típico: 10 dígitos → +52 + 10
  if (strlen($digits) === 10) {
    $e164 = $defaultCountry . $digits;
    $e164 = preg_replace('/[^0-9\+]/', '', $e164) ?? '';
    if ($e164 !== '' && $e164[0] !== '+') $e164 = '+' . $e164;
    return 'whatsapp:' . $e164;
  }

  // 12 dígitos iniciando con 52 (MX) → +52 + 10
  if (strlen($digits) === 12 && strpos($digits, '52') === 0) {
    return 'whatsapp:+' . $digits;
  }

  // E164 sin + (varios países) → agregar +
  if (strlen($digits) >= 11 && strlen($digits) <= 15) {
    return 'whatsapp:+' . $digits;
  }

  return '';
}

// ====== Cargar configuración ======
$configFile = __DIR__ . '/includes/config.local.php';
if (!is_file($configFile)) fail("NO_CONFIG", "Falta includes/config.local.php", 500);
$cfg = require $configFile;

$tw = $cfg['twilio'] ?? [];
if (empty($tw['enabled'])) fail("TWILIO_DISABLED", "twilio.enabled=false");

$sid   = trim((string)($tw['sid'] ?? ''));
$token = trim((string)($tw['token'] ?? ''));
$from  = trim((string)($tw['wa_from'] ?? ''));
$defaultCountry = (string)($tw['default_country'] ?? '+52');

if ($sid === '' || $token === '' || $from === '') {
  fail("TWILIO_CONFIG", "Config incompleta. Revisa twilio.sid, twilio.token, twilio.wa_from", 500);
}

// Normalizar FROM
$from = preg_replace('/\s+/', '', $from) ?? '';
if (stripos($from, 'whatsapp:') !== 0) $from = 'whatsapp:' . ltrim($from);
if (stripos($from, 'whatsapp:+') !== 0) {
  $from = 'whatsapp:+' . ltrim(substr($from, 9), '+');
}

// ====== Inputs ======
$rawTo = (string)($_GET['to'] ?? '');
$msg   = trim((string)($_GET['msg'] ?? 'Prueba WhatsApp Ventas Blanquita'));

if ($rawTo === '') {
  fail("NO_TO", "Falta parámetro ?to= (ej: whatsapp:+524441234567 o +524441234567 o 4441234567)");
}

$to = normalize_to_whatsapp($rawTo, $defaultCountry);
if ($to === '') {
  fail("BAD_TO", "No se pudo normalizar el número. Recibí: {$rawTo}. Ejemplo válido: whatsapp:+524441234567");
}

// ====== Parche Twilio antes del autoload ======
$poly = __DIR__ . '/includes/twilio_polyfill.php';
if (is_file($poly)) require_once $poly;

// ====== Autoload Twilio ======
$autoloadCandidates = [];
if (!empty($cfg['vendor_autoload'])) $autoloadCandidates[] = $cfg['vendor_autoload'];
$autoloadCandidates[] = __DIR__ . '/vendor/autoload.php';
$autoloadCandidates[] = dirname(__DIR__) . '/vendor/autoload.php';

$autoload = null;
foreach ($autoloadCandidates as $f) {
  if (is_string($f) && $f !== '' && is_file($f)) { $autoload = $f; break; }
}
if (!$autoload) {
  fail("NO_AUTOLOAD", "No encontré Composer autoload. Define 'vendor_autoload' en config.local.php o instala vendor/.", 500);
}
require_once $autoload;

try {
  if (!class_exists(\Twilio\Rest\Client::class)) {
    fail("NO_TWILIO_SDK", "No está disponible Twilio SDK (twilio/sdk). Revisa composer/vendor.", 500);
  }

  $client = new \Twilio\Rest\Client($sid, $token);
  $res = $client->messages->create($to, [
    "from" => $from,
    "body" => $msg,
  ]);

  echo json_encode([
    "ok" => true,
    "to_raw" => $rawTo,
    "to_norm" => $to,
    "from_norm" => $from,
    "sid" => $res->sid ?? null,
    "status" => $res->status ?? null
  ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

} catch (Throwable $e) {
  fail("EXCEPTION", $e->getMessage(), 500);
}
